(($) => {

  const {
    input,
    textarea,
    select,
    dialog,
    errorDialog,
    inputRepeaterWidget,
    modal,
    betterModal,
    icons,
    mediaPicker,
    toggle,
    moreMenu,
    tinymceElement,
    tooltipIcon,
    tooltip,
    clickedIn,
  } = HollerBox.elements

  const { sprintf, __, _x, _n } = wp.i18n

  const Keywords = HollerBox._editor.Keywords
  const Controls = HollerBox._editor.Controls
  const Integrations = HollerBox._editor.Integrations
  const Templates = HollerBox._editor.Templates
  const AdvancedDisplayRules = HollerBox._editor.AdvancedDisplayRules
  const Triggers = HollerBox._editor.Triggers

  const { Div, Input, Select, Button, Span, Label, Textarea, Fragment } = MakeEl

  const morph = (selector, children, childrenOnly = true ) => {
    morphdom(document.querySelector(selector), Div({}, children), {
      childrenOnly,
    })
  }

  const {
    singleControl,
    itemPicker,
    apiGet,
    apiPost,
  } = HollerBox._helpers

  const singleControlEl = ({
    label = '',
    control = null,
    hidden = false,
    stacked = false,
  }) => {

    return Div({
      className: `control ${stacked ? 'stacked' : ''} ${hidden ? 'hidden' : ''}`,
    }, [
      `<label>${label}</label>`,
      control,
    ])
  }

  const IntegrationIcons = {
    //language=HTML
    keap: `
		<svg viewBox="0 0 500 500" xmlns="http://www.w3.org/2000/svg">
			<path fill="#36a635"
			      d="M 375.164 405.564 L 340.103 441.896 C 335.017 448.072 327.387 446.982 321.211 441.896 L 197.319 315.461 L 197.319 433.176 C 197.319 440.624 192.414 445.71 183.695 445.71 L 133.557 445.71 C 126.108 445.71 121.022 440.624 121.022 432.994 L 121.022 66.587 C 121.022 58.957 126.108 53.871 133.557 53.871 L 183.695 53.871 C 192.414 53.871 197.319 58.957 197.319 66.406 L 197.319 186.482 L 366.262 186.482 C 373.892 186.482 378.978 191.569 378.978 199.199 L 378.978 250.427 C 378.978 257.875 373.892 262.961 366.262 262.961 L 251.272 262.961 L 375.164 388.124 C 380.068 393.029 380.068 399.387 375.164 405.564 Z"
			      style=""/>
		</svg>`,
    //language=HTML
    mailpoet: `
		<svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 152 156.4">
			<path fill="#fe5301"
			      d="M37.7 89.1c3.5 0 6-.8 7.2-2.3a8 8 0 0 0 2-5.4V35.7l17 45.1a12.7 12.7 0 0 0 3.7 5.4c1.6 1.3 4 2 7.2 2a12.5 12.5 0 0 0 6-1.4 8.4 8.4 0 0 0 3.8-5l18.1-50V81a8.5 8.5 0 0 0 2.1 6.1c1.4 1.4 3.7 2.2 7 2.2 3.4 0 5.8-.8 7.1-2.3a8 8 0 0 0 2-5.4V8.7a7.5 7.5 0 0 0-3.3-6.6c-2-1.4-5-2.1-8.6-2.1a19.3 19.3 0 0 0-9.4 2 11.6 11.6 0 0 0-5 6.8L74.8 67.1 54.4 8.4A12.4 12.4 0 0 0 50 2.2C48 .7 45 0 41.1 0a16.5 16.5 0 0 0-8.9 2.1 8.2 8.2 0 0 0-3.5 7.2v71.5c0 2.8.7 4.8 2 6.2 1.5 1.4 3.7 2.1 7 2.1ZM149 116.6l-2.4-1.9a7.4 7.4 0 0 0-9.4.3 19.6 19.6 0 0 1-12.5 4.6h-21.4A37 37 0 0 0 77 130.5l-1.1 1.2-1.1-1.1a37.3 37.3 0 0 0-26.3-10.9H27a19.6 19.6 0 0 1-12.4-4.6 7.3 7.3 0 0 0-9.4-.3l-2.4 1.9a7.4 7.4 0 0 0-2.8 5.5 7.1 7.1 0 0 0 2.4 5.7 37.3 37.3 0 0 0 24.6 9.5h21.6a19.6 19.6 0 0 1 18.9 14.4v.2c.1.7 1.2 4.4 8.5 4.4s8.4-3.7 8.5-4.4v-.2a19.6 19.6 0 0 1 18.9-14.4H125a37.3 37.3 0 0 0 24.6-9.5 7.4 7.4 0 0 0 2.4-5.7 7.9 7.9 0 0 0-3-5.6Z"
			      data-name="Layer 1"/>
		</svg>`,
    //language=HTML
    convertkit: `
		<svg xmlns="http://www.w3.org/2000/svg" fill="none"
		     viewBox="0 0 172 160">
			<path fill="#FB6970"
			      d="M82.7 126.3a51.3 51.3 0 0 0 52.8-50.5c0-26.2-21.6-42.1-36-42.1-19.9 0-35.9 14-38.1 35-.5 3.8-3.5 7-7.4 7H32.4a4.5 4.5 0 0 1-4.6-4.7c.9-18 7-35 18.4-48A69.5 69.5 0 0 1 99.6 0c37 0 71.8 33.8 71.8 75.8 0 46.5-38.3 84.2-87.9 84.2-35.4 0-71-23.3-83.5-55.8v-.4l.8-2.7c.5-1.2 1-2.4.9-3.9l-.2-4.5a5.5 5.5 0 0 1 3.2-5.3l1.8-.8a7 7 0 0 0 3.9-5 7 7 0 0 1 6.9-5.8c5.2 0 9.8 3.5 11.5 8.5 8.4 24.2 21.6 42 54 42Z"/>
		</svg>`,
    //language=HTML
    mailchimp: `
		<svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 24 24">
			<path
				d="M18 11.3h.5v-.9c-.1-.6-.3-1-.6-.9-.3 0-.4.5-.2 1 0 .4.1.7.3.8zm-2.9.5c.3.1.4.2.5.1v-.2l-.7-.5a1.7 1.7 0 0 0-1.6.2c-.2.1-.3.3-.3.4l1.2-.2c.4 0 .7 0 1 .2zm-.5.3-.6.2-.1.3h.3a2 2 0 0 1 1-.1h.3v-.2c-.1 0-.4-.3-.9-.2zm2.4 1 .5-.1c.1-.2 0-.4-.2-.5l-.5.1c-.1.2 0 .4.2.5zm1.4-1.2c-.2 0-.3.2-.3.4s0 .4.3.4c.1 0 .3-.1.3-.4 0-.2-.2-.4-.3-.4zM6.9 16h-.2a.6.6 0 0 1-.2 0 .3.3 0 0 1-.3-.1v-.6c.2-.3.5-.8.2-1.3-.2-.4-.5-.6-.9-.7a1.1 1.1 0 0 0-1 .4c-.4.4-.4 1-.3 1.2l.3-.1c0-.2 0-.3.2-.5a.7.7 0 0 1 .9-.2c.3.2.4.5.3.9l-.2.7c0 .5.4.7.7.7l.5-.2V16zm15.8-1-.2-.5-.2-.5c.4-.6.4-1 .4-1.4 0-.3-.2-.6-.5-1a4 4 0 0 0-1.8-.8l-.4-.1V9.3L19.6 8c-.2-.7-.5-1.3-.9-1.6 1.1-1.2 1.8-2.5 1.8-3.6 0-2-2.5-2.7-5.7-1.4l-.7.3A666.1 666.1 0 0 0 13 .5C9.4-2.6-2 9.9 1.7 13l.8.6a3.9 3.9 0 0 0-.2 1.8c0 .9.5 1.7 1.2 2.3.7.6 1.5 1 2.4 1a9.1 9.1 0 0 0 8.3 5.3c4 .1 7.5-1.8 8.9-5.2 0-.2.5-1.3.5-2.3 0-1-.6-1.3-1-1.3zM6.2 17.8a2 2 0 0 1-.4 0c-1.2 0-2.5-1-2.7-2.4-.1-1.4.6-2.5 2-2.8a2.2 2.2 0 0 1 .5 0c.7 0 1.8.6 2 2.2.3 1.4 0 2.8-1.4 3zm-1.4-6c-.8.1-1.5.6-2 1.2l-.8-.8c-.7-1.3.8-3.9 1.8-5.3 2.5-3.7 6.5-6.5 8.3-6 .3.1 1.3 1.3 1.3 1.3s-1.9 1-3.5 2.4a16 16 0 0 0-5 7zm12.8 5.5V17s-2 .3-3.7-.4c.2-.6.7-.4 1.5-.3A11 11 0 0 0 19 16c.8-.3 2-.7 2.7-1.4.3.6.4 1.3.4 1.3h.4c.2.2.3.4.2 1a5 5 0 0 1-1.4 2.7 5.7 5.7 0 0 1-1.6 1.2 7 7 0 0 1-1.1.5c-2.9 1-5.8-.1-6.7-2.3a3.6 3.6 0 0 1-.2-.5c-.4-1.5 0-3.2 1-4.3l.1-.3v-.2c-.5-.6-1.7-1.5-1.5-3.3.2-1.3 1.3-2.2 2.4-2.1h.2l1.3.1c.6 0 1.1 0 1.8-.6.2-.2.4-.4.7-.4h.3a.9.9 0 0 1 .4.1c.6.4.6 1.2.7 1.8v1.5c0 .6.2.7.5.8l.6.1a3 3 0 0 1 1.4.7l.2.5c.1.6-.4 1.4-1.9 2-1.6.8-3.6 1-4.9.8h-.5c-1-.2-1.7 1.2-1 2.2.4.6 1.5 1 2.7 1 2.6 0 4.6-1.1 5.3-2v-.2c.1 0 0 0 0 0-.6.4-3.3 2-6.2 1.5l-.6-.1c-.3-.1-.8-.4-.9-1 2.3.8 3.8.1 3.8.1zm-3.7-.4zm-4.4-10c.9-1.1 2-2 3-2.5l-.3.7c.7-.3 1.8-.8 2.7-.9v.1l-.4.4v.1c.7 0 1.6.2 2.2.6a8.9 8.9 0 0 0-7.2 1.5z"/>
		</svg>`,
    //language=HTML
    hubspot: `
		<svg xmlns="http://www.w3.org/2000/svg" xml:space="preserve"
		     viewBox="0 0 512 512">
  <path fill="#ff7a59"
        d="M266.197 216.109c-22.551 21.293-36.655 51.48-36.655 84.991 0 26.326 8.714 50.582 23.359 70.08l-44.473 44.74c-3.953-1.438-8.176-2.245-12.579-2.245-9.702 0-18.776 3.774-25.605 10.602-6.828 6.827-10.602 15.989-10.602 25.696 0 9.701 3.773 18.775 10.602 25.605 6.829 6.826 15.993 10.42 25.605 10.42 9.703 0 18.777-3.505 25.695-10.42a36.103 36.103 0 0 0 10.602-25.605c0-3.774-.538-7.369-1.707-10.873l44.923-45.102c19.765 15.183 44.381 24.169 71.244 24.169 64.599 0 116.797-52.38 116.797-116.977 0-58.578-42.854-107.093-99.007-115.628v-55.343c15.723-6.65 25.335-21.384 25.335-38.545 0-23.449-18.777-43.034-42.227-43.034-23.448 0-41.956 19.585-41.956 43.034 0 17.161 9.613 31.895 25.335 38.545v54.983c-13.655 1.887-26.593 6.019-38.362 12.219-24.796-18.778-105.565-76.997-151.746-112.126 1.078-3.953 1.798-8.085 1.798-12.397C142.573 47.023 121.46 26 95.495 26 69.62 26 48.597 47.023 48.597 72.898c0 25.965 21.023 46.988 46.898 46.988 8.805 0 16.98-2.606 24.078-6.828l146.624 103.051zm80.409 146.986c-34.229 0-61.991-27.763-61.991-61.994 0-34.229 27.762-61.99 61.991-61.99 34.23 0 61.992 27.761 61.992 61.99.001 34.231-27.761 61.994-61.992 61.994z"/>
</svg>`,
    //language=HTML
    activecampaign: `
		<svg xmlns="http://www.w3.org/2000/svg"
		     xmlns:xlink="http://www.w3.org/1999/xlink" x="0px" y="0px"
		     viewBox="0 0 288 288" xml:space="preserve">
    <path fill="#356AE6" class="st0" d="M146.6,150.7c2.6,0,5.2-1,8-3.1c3.3-2.3,6.2-4.2,6.2-4.2l1-0.7l-1-0.7c-0.4-0.3-43.9-30.5-48.4-33.4
		c-2.1-1.5-4.5-1.8-6.4-0.9c-1.8,0.9-2.8,2.8-2.8,5.1v10.3l0.4,0.2c0.3,0.2,30.2,21,36,24.9C142,149.9,144.3,150.7,146.6,150.7z"/>
			<path fill="#356AE6" class="st0" d="M204.9,132.6c-2.6-1.9-95.9-67-99.9-69.8l-1.3-0.9v16c0,5.1,2.7,7,6,9.4c0,0,71.6,49.9,80.5,56
		c-8.9,6.2-76.3,52.9-80.6,55.6c-5.1,3.4-5.6,5.6-5.6,10.2v17.2c0,0,98.7-70.6,100.8-72.2l0,0c4.5-3.4,5.5-7.4,5.6-10.3l0-1.8
		C210.5,138.4,208.6,135.2,204.9,132.6z"/>
</svg>`,
    //language=HTML
    drip: `
		<svg xmlns="http://www.w3.org/2000/svg" xml:space="preserve"
		     viewBox="0 0 314.3 418.5">
  <path fill="#f224f2"
        d="M314.2 253.4H267l.2 4.9c0 69.8-57.2 113-110 113a111.7 111.7 0 0 1-110-117.9H.1l-.1 4.9a159 159 0 0 0 157.1 160.2 159 159 0 0 0 157.1-165M157.1 59.9l33.7 45.4h59.4C206.8 45.5 157.2 0 157.2 0S107.4 45.5 64 105.3h59.4L157 59.9m31.6 120.5a31.4 31.4 0 1 1-62.9 0 31.4 31.4 0 0 1 62.9 0m94.3 0a31.4 31.4 0 1 1-62.9 0 31.4 31.4 0 0 1 62.9 0m-188.6 0a31.4 31.4 0 1 1-62.9 0 31.4 31.4 0 0 1 62.9 0"/>
</svg>`,
  }

  const FomoIntegrations = {
    edd: __('Easy Digital Downloads'),
    woo: __('WooCommerce'),
    give: __('GiveWP'),
    pmp: __('Paid Memberships Pro'),
    memberpress: __('MemberPress'),
  }

  let defaultFomoIntegration = false

  for (let integration in FomoIntegrations) {
    if (HollerBoxPro.installed[integration]) {
      defaultFomoIntegration = integration
      break
    }
  }

  function stripHTML (html) {
    let tmp = document.createElement('DIV')
    tmp.innerHTML = html
    return tmp.textContent || tmp.innerText || ''
  }

  const ProControls = {
    position_banner: {
      name: __('Position', 'holler-box'),
      render: ({
        position = 'top-left',
      }) => {
        return [
          singleControl({
            label: __('Position'),
            control: select({
              id: 'position',
            }, {
              'top-left': __('Top'),
              'bottom-left': __('Bottom'),
            }, position),
          }),
        ].join('')
      },
      onMount: (settings, updateSetting) => {
        $('#position').on('change', e => {
          updateSetting({
            position: e.target.value,
          }, {
            suppressAnimations: false,
          })
        })
      },
    },
    position_sidebar: {
      name: __('Position', 'holler-box'),
      render: ({
        position = 'center-right',
      }) => {
        return [
          singleControl({
            label: __('Position'),
            control: select({
              id: 'position',
            }, {
              'center-left': __('Left'),
              'center-right': __('Right'),
            }, position),
          }),
        ].join('')
      },
      onMount: (settings, updateSetting) => {
        $('#position').on('change', e => {
          updateSetting({
            position: e.target.value,
          }, {
            suppressAnimations: false,
          })
        })
      },
    },
    fomo: {
      name: __('Fomo', 'holler-box'),
      render: ({
        fomo_integration = '',
        fomo_text = '{{name}} purchased {{product}}',
        fomo_image = 'avatar',
        fomo_time_ago = 14,
        fomo_display_time = 5,
        fomo_loop_delay = 4,
      }) => {

        return [
          singleControl({
            label: __('Integration'),
            control: select({
              id: 'fomo_integration',
            }, {
              '': __('Select one'),
              ...FomoIntegrations,
            }, fomo_integration),
          }),
          singleControl({
            label: __('Show sales within'),
            control: `<span>${input({
              id: 'fomo_time_ago',
              value: fomo_time_ago,
              type: 'number',
              style: {
                width: 50,
              },
            })} ${__('days')}</span>`,
          }),
          `<div class="control"><div>`,
          `<p><label for="fomo-text">${__('Message')}</label></p>`,
          input({
            id: 'fomo_text',
            value: fomo_text,
            className: 'full-width',
            placeholder: '{{name}} purchased {{product}}',
          }),
          `<p>${sprintf(__('Use %s and %s to personalize the message.'),
            `<code>{{name}}</code>`,
            `<code>{{product}}</code>`)}</p>`,
          `</div></div>`,
          singleControl({
            label: __('Image'),
            control: select({
              id: 'fomo_image',
            }, {
              product: __('Product Image'),
              avatar: __('User Avatar'),
            }, fomo_image),
          }),
          singleControl({
            label: __('Show for'),
            control: `<span>${input({
              id: 'fomo_display_time',
              value: fomo_display_time,
              type: 'number',
              style: {
                width: 50,
              },
            })} ${__('seconds')}</span>`,
          }),
          singleControl({
            label: __('Hide for'),
            control: `<span>${input({
              id: 'fomo_loop_delay',
              value: fomo_loop_delay,
              type: 'number',
              style: {
                width: 50,
              },
            })} ${__('seconds')}</span>`,
          }),
        ].join('')
      },
      onMount: (settings, updateSetting) => {
        $('#fomo_text,#fomo_display_time,#fomo_loop_delay,#fomo_time_ago,#fomo_integration,#fomo_image').
        on('change', e => {
          updateSetting({
            [e.target.id]: e.target.value,
          }, {
            suppressAnimations: false,
          })
        })
      },
    },
    yesNoButton: {
      name: __('Buttons', 'holler-box'),
      render: ({
        button_alignment = 'vertical',
        button_swap_order = false,
      }) => {
        return [
          singleControl({
            label: __('Alignment'),
            control: select({
              id: 'button_alignment',
            }, {
              horizontal: __('Horizontal'),
              vertical: __('Vertical'),
            }, button_alignment),
          }),
          singleControl({
            hidden: button_alignment === 'vertical',
            label: __('Swap order'),
            control: toggle({
              id: 'button_swap_order',
              checked: button_swap_order,
            }),
          }),
        ].join('')
      },
      onMount: (settings, updateSetting) => {
        $('#button_alignment').on('change input', e => {
          updateSetting({
            [e.target.id]: e.target.value,
          }, {
            morph: true,
          })
        })

        $('#button_swap_order').on('change', e => {
          updateSetting({
            [e.target.id]: e.target.checked,
          })
        })
      },
      css: ({ id, button_swap_order = false }) => {

        let rules = []

        if (button_swap_order) {
          // language=CSS
          rules.push(`
              #${id} .yes-no-buttons.horizontal {
                  flex-direction: row-reverse;
              }
          `)
        }

        return rules.join('')

      },
    },
    choseYes: {
      name: __('Yes Button', 'holler-box'),
      render: ({
        after_submit = 'close',
        redirect_url = '',
        success_message = '',
        yes_text = 'Yes!',
        yes_color = '',
        yes_text_color = '',
      }) => {
        let controls = [
          singleControl({
            label: __('Button Text'),
            control: input({
              id: 'yes_text',
              value: yes_text,
            }),
          }),
          singleControl({
            label: __('Button Color'),
            control: input({
              id: 'yes_color',
              value: yes_color,
            }),
          }),
          singleControl({
            label: __('Text Color'),
            control: input({
              id: 'yes_text_color',
              value: yes_text_color,
            }),
          }),
          '<hr/>',
          singleControl({
            label: __('What happens when <b>Yes</b> is chosen?'),
            control: select({
              id: 'after-submit',
            }, {
              message: __('Show a message'),
              redirect: __('Redirect to page'),
              close: __('Close the popup'),
            }, after_submit),
          }),
          `<div id="dependent-controls"></div>`,
        ]

        return controls.join('')
      },
      onMount: (popup, updateSetting) => {
        Controls.submit.onMount(popup, updateSetting);

        [
          'yes_text_color',
          'yes_color',
        ].forEach(id => {
          $(`#${id}`).wpColorPicker({
            change: (e, ui) => {
              updateSetting({
                [id]: ui.color.toString(),
              })
            },
          })
        })

        $('#yes_text').on('change input', e => {
          updateSetting({
            [e.target.id]: e.target.value,
          })
        })
      },
      css: ({
        id,
        yes_color = '',
        yes_text_color = '',
      }) => {

        // language=CSS
        return `
            #${id} .holler-box-button.choose-yes {
                background-color: ${yes_color};
                color: ${yes_text_color};
            }
        `
      },
    },
    choseNo: {
      name: __('No Button', 'holler-box'),
      render: ({
        no_chosen = 'close',
        no_redirect_url = '',
        no_message = '',
        no_text = 'No!',
        no_color = '',
        no_text_color = '',
      }) => {
        let controls = [
          singleControl({
            label: __('Button Text'),
            control: input({
              id: 'no_text',
              value: no_text,
            }),
          }),
          singleControl({
            label: __('Button Color'),
            control: input({
              id: 'no_color',
              value: no_color,
            }),
          }),
          singleControl({
            label: __('Text Color'),
            control: input({
              id: 'no_text_color',
              value: no_text_color,
            }),
          }),
          '<hr/>',
          singleControl({
            label: __('What happens when <b>No</b> is chosen?'),
            control: select({
              id: 'no-chosen',
            }, {
              message: __('Show a message'),
              redirect: __('Redirect to page'),
              close: __('Close the popup'),
            }, no_chosen),
          }),
          `<div id="no-dependent-controls"></div>`,
        ]

        return controls.join('')
      },
      onMount: ({
        no_chosen = 'close',
        no_redirect_url = '',
        no_message = '',
      }, updateSetting) => {

        [
          'no_text_color',
          'no_color',
        ].forEach(id => {
          $(`#${id}`).wpColorPicker({
            change: (e, ui) => {
              updateSetting({
                [id]: ui.color.toString(),
              })
            },
          })
        })

        $('#no_text').on('change input', e => {
          updateSetting({
            [e.target.id]: e.target.value,
          })
        })

        const mountDependentControls = () => {

          const setUI = (ui) => {
            $('#no-dependent-controls').html(ui)
          }

          switch (no_chosen) {
            case 'message':
              setUI([
                textarea({
                  id: 'no-message',
                  value: no_message,
                }),
                `<p class="holler-notice info">${__(
                  'Shortcodes will be rendered on the frontend.')}</p>`,
              ].join(''))

              wp.editor.remove('no-message')
              tinymceElement('no-message', {
                tinymce: {
                  content_style: HollerBox.editor.tinymceCSS(),
                },
              }, (value) => {
                no_message = value
                updateSetting({
                  no_message: value,
                })
              })

              break

            case 'redirect':
              setUI(input({
                type: 'url',
                id: 'no_redirect_url',
                className: 'full-width',
                placeholder: 'https://example.com',
                value: no_redirect_url,
              }))

              $('#no_redirect_url').on('input change', e => updateSetting({
                no_redirect_url: e.target.value,
              }))

              break
            default:
            case 'close':
              setUI('')
              break
          }
        }

        $('#no-chosen').on('change', e => {

          no_chosen = e.target.value
          updateSetting({
            no_chosen,
          })

          mountDependentControls()
        })

        mountDependentControls()
      },
      css: ({
        id,
        no_color = '',
        no_text_color = '',
      }) => {

        // language=CSS
        return `
            #${id} .holler-box-button.choose-no {
                background-color: ${no_color};
                color: ${no_text_color};
            }
        `
      },
    },
    survey: {
      name: __('Survey Questions'),
      render: ({
        questions = [],
        allow_skipping = false,
      }) => {
        return [
          `<div id="questions"></div>`,
          `<button class="holler-button secondary" id="add-survey-question">${__(
            'Add Question')}</button>`,
          `<hr/>`,
          singleControl({
            label: __('Allow skipping?'),
            control: toggle({
              id: 'allow-skipping',
              checked: allow_skipping,
            }),
          }),
        ].join('')
      },
      onMount: (settings, updateSetting) => {

        let {
          questions = [],
        } = settings

        $('#allow-skipping').on('change', e => {
          updateSetting({
            allow_skipping: e.target.checked,
          })
        })

        const __mount = () => {

          let $questions = $('#questions')

          questions = questions.filter(q => q !== null)

          $questions.html(
            questions.map((question, i) =>
              // language=HTML
              `
				  <div class="question" data-id="${i}">
					  <div class="name">${stripHTML(question.prompt)}</div>
					  <button
						  class="holler-button secondary text icon question-more">
						  ${icons.verticalDots}
					  </button>
				  </div>`).join(''))

          $questions.find('.question').on('click', e => {

            let questionId = parseInt(e.currentTarget.dataset.id)

            if (clickedIn(e, '.question-more')) {
              moreMenu($(e.currentTarget).find('.question-more'), {
                items: [
                  {
                    key: 'edit',
                    text: __('Edit'),
                  },
                  {
                    key: 'delete',
                    text: `<span class="holler-text danger">${__(
                      'Delete')}</span>`,
                  },
                ],
                onSelect: k => {
                  switch (k) {
                    case 'edit':
                      editQuestionModal(questionId)
                      break
                    case 'delete':

                      questions = questions.filter((q, i) => i !== questionId)

                      updateSetting({
                        questions,
                      })

                      __mount()

                      break
                  }
                },
              })
              return
            }

            editQuestionModal(questionId)
          })

          $questions.sortable({
            update: (e, ui) => {

              let newOrder = []

              $questions.find('.question').each((i, el) => {
                newOrder.push(questions[parseInt(el.dataset.id)])
              })

              questions = newOrder

              updateSetting({
                questions,
              })

              __mount()

            },
          })
        }

        const __updateQuestion = (id, props) => {
          questions[id] = {
            ...questions[id],
            ...props,
          }

          updateSetting({
            questions,
          }, {
            overrides: {
              current: id,
              __state: 'survey',
            },
          })

          __mount()
        }

        const __addQuestion = () => {
          questions.push({
            prompt: '',
            choices: [],
            type: 'choices',
          })

          editQuestionModal(questions.length - 1)
        }

        const editQuestionModal = (id) => {

          let {
            type = 'choices',
            prompt = '',
          } = questions[id]

          modal({
            dialogClasses: 'has-header',
            // language=HTML
            content: `
				<div class="holler-header">
					<h3>${sprintf(__('Edit Question #%s'), id + 1)}</h3>
					<button
						class="holler-button secondary text icon holler-modal-button-close"><span
						class="dashicons dashicons-no-alt"></span>
					</button>
				</div>
				<div class="holler-rows-and-columns">
					<div class="row">
						<div class="col">
							${textarea({
								id: 'prompt',
								value: prompt,
							})}
						</div>
					</div>
					<div class="row">
						<div class="col">
							<label>${__('Answer Type')}</label>
							${select({
								id: 'question-type',
							}, {
								choices: __('Choices'),
								text: __('Short text'),
								textarea: __('Paragraph text'),
								yes_no: __('Yes/No'),
							}, type)}
						</div>
						<div class="col"></div>
					</div>
					<div class="row">
						<div class="col">
							<div id="question-type-settings"></div>
						</div>
					</div>
				</div>
            `,
            onOpen: () => {

              wp.editor.remove('prompt')

              tinymceElement('prompt', {
                tinymce: {
                  content_style: HollerBox.editor.tinymceCSS(),
                },
              }, (prompt) => {
                __updateQuestion(id, {
                  prompt,
                })
              })

              $('#question-type').on('change', e => {
                type = e.target.value
                __updateQuestion(id, { type })
                __mountTypeSettings()
              })

              const __mountTypeSettings = () => {

                const __setSettings = (html) => {
                  $('#question-type-settings').html(html)
                }

                switch (type) {
                  case 'text':
                  case 'textarea':

                    // language=HTML
                    __setSettings(`
						<div class="holler-rows-and-columns">
							<div class="row">
								<div class="col">
									<label>${__('Placeholder text')}</label>
									${input({
										id: 'question-placeholder',
										value: questions[id].placeholder,
									})}
								</div>
								<div class="col">
									<label>${__('Next question text')}</label>
									${input({
										id: 'next-text',
										value: questions[id].next,
									})}
								</div>
							</div>
						</div>`)

                    $('#question-placeholder').on('input', e => {
                      __updateQuestion(id, {
                        placeholder: e.target.value,
                      })
                    })

                    $('#next-text').on('input', e => {
                      __updateQuestion(id, {
                        next: e.target.value,
                      })
                    })

                    break
                  case 'choices':
                    // language=HTML
                    __setSettings(`
						<div class="display-flex gap-10">${__(
							'Labels & values are different')} ${toggle({
							id: 'custom-values',
							checked: questions[id].custom_values ?? false,
						})} ${tooltipIcon('custom-values-help')}
						</div>
						<div id="question-choices"></div>
                    `)

                    tooltip('#custom-values-help', {
                      content: __(
                        'Use a custom value instead of the choice name. If no value is set, the choice name will be used instead.'),
                    })

                    $('#custom-values').on('change', e => {
                      __updateQuestion(id, {
                        custom_values: e.target.checked,
                      })

                      __mountTypeSettings()
                    })

                    let callbacks = [
                      props => input({
                        ...props,
                        placeholder: __('Choice Name'),
                      }),
                    ]

                    if (questions[id].custom_values) {
                      callbacks.push(props => input({
                        ...props,
                        placeholder: __('Custom Value'),
                      }))
                    }

                    inputRepeaterWidget('#question-choices', {
                      rows: questions[id].choices.map(c => Array.isArray(c) ? c : [c]),
                      sortable: true,
                      cellCallbacks: callbacks,
                      onChange: (rows) => {
                        __updateQuestion(id, {
                          choices: rows,
                        })
                      },
                    }).mount()

                    break
                }
              }

              __mountTypeSettings()

            },

          })

        }

        $('#add-survey-question').on('click', e => {
          __addQuestion()
          __mount()
        })

        __mount()

      },
      css: () => '',
    },
    survey_links: {
      name: __('Survey Choices'),
      render: () => {
        return `<div id="survey-links"></div>`
      },
      onMount: ({ choices = [] }, updateSetting) => {

        const __morph = () => morph('#survey-links', render())

        const __addChoice = () => {
          choices.push({
            choice: 'New choice',
          })

          updateSetting({
            choices,
          })

          __morph()
        }

        const __updateChoice = (id, stuff) => {
          choices[id] = {
            ...choices[id],
            ...stuff,
          }

          updateSetting({ choices })

          __morph()
        }

        const __deleteChoice = (id) => {
          choices.splice(id, 1)
          updateSetting({ choices })

          __morph()
        }

        const __editQuestion = (id) => {
          betterModal({
            dialogClasses: 'has-header',
            width: 500,
            // language=HTML
            render: () => `
				<div class="holler-header">
					<h3>${sprintf(__('Edit Choice #%s'), id + 1)}</h3>
					<button
						class="holler-button secondary text icon holler-modal-button-close"><span
						class="dashicons dashicons-no-alt"></span>
					</button>
				</div>
				<div id="question-settings"></div>
            `,
            onMount: ({ close, onMount }) => {

              const ___morph = () => {
                morph('#question-settings', __render())
                if (choices[id].result === 'message') {
                  wp.editor.remove('choice-message')
                  tinymceElement('choice-message', {
                    tinymce: {
                      content_style: HollerBox.editor.tinymceCSS(),
                    },
                  }, message => {
                    __updateChoice(id, { message })
                  })
                }
              }

              const __result = () => {
                switch (choices[id].result) {
                  case 'redirect':
                    return Div({
                      className: 'display-flex column gap-10',
                    }, [
                      Label({}, 'Redirect to...'),
                      Input({
                        id: 'choice-redirect',
                        type: 'url',
                        name: 'choice_redirect',
                        placeholder: HollerBox.home_url,
                        value: choices[id].redirect ?? '',
                        onInput: e => __updateChoice(id, { redirect: e.target.value }),
                      }),
                    ])
                  case 'message':
                    return Textarea({
                      id: 'choice-message',
                      value: choices[id].message ?? '',
                    })
                  default:
                  case 'close':
                    return null
                }

              }

              const __render = () => Div({
                className: 'display-flex column gap-10',
              }, [
                Div({
                  className: 'display-flex column gap-10',
                }, [
                  Label({}, 'Choice text'),
                  Input({
                    id: 'choice-text',
                    name: 'choice_text',
                    placeholder: 'Choice text...',
                    value: choices[id].choice,
                    onInput: e => __updateChoice(id, { choice: e.target.value }),
                  }),
                ]),
                Div({
                  className: 'display-flex column gap-10',
                }, [
                  Label({}, 'When chosen...'),
                  Select({
                    id: 'choice-result',
                    name: 'choice_result',
                    selected: choices[id].result ?? 'close',
                    options: {
                      redirect: 'Redirect to a page',
                      message: 'Show a message',
                      close: 'Close the popup',
                    },
                    onChange: e => {
                      __updateChoice(id, { result: e.target.value })
                      ___morph()
                    },
                  }),
                ]),
                __result(),
              ])

              ___morph()

            },

          })

        }

        const render = () => Div({
          id: 'questions',
          onCreate: el => {

            let $questions = $(el)

            $questions.sortable({
              update: (e, ui) => {

                let newOrder = []

                $questions.find('.question').each((i, el) => {
                  newOrder.push(choices[parseInt(el.dataset.id)])
                })

                choices = newOrder

                updateSetting({
                  choices,
                })

                __morph()

              },
            })
          },
        }, [
          ...choices.map((a, i) => Div({
            className: 'question',
            dataId: i,
            id: `question-${i}`,
            onClick: e => {

              if ( clickedIn(e, 'button.question-more') ){
                return
              }

              __editQuestion(i)
            },
          }, [
            Div({
              className: 'name',
            }, a.choice),
            Button({
              id: `question-${i}-more`,
              className: 'holler-button secondary text icon question-more',
              onClick: e => {
                moreMenu(e.currentTarget, {
                  items: [
                    {
                      key: 'edit',
                      text: __('Edit'),
                    },
                    {
                      key: 'delete',
                      text: `<span class="holler-text danger">${__('Delete')}</span>`,
                    },
                  ],
                  onSelect: k => {
                    switch (k) {
                      case 'edit':
                        __editQuestion(i)
                        break
                      case 'delete':
                        __deleteChoice(i)
                        break
                    }
                  },
                })
              },
            }, icons.verticalDots),
          ])),
          Button({
            className: 'holler-button secondary',
            onClick: __addChoice,
          }, 'Add Choice'),
        ])

        __morph()
      },
    },
    form_prompt: {
      name: __('Form Prompt', 'holler-box'),
      render: ({
        form_prompt = '',
      }) => {
        return [
          textarea({
            id: 'form-prompt',
            value: form_prompt,
          }),
          `<p class="holler-notice info">${__(
            'Shortcodes will be not rendered here.')}</p>`,
        ].join('')
      },
      onMount: (settings, updateSetting) => {

        wp.editor.remove('form-prompt')

        tinymceElement('form-prompt', {
          tinymce: {
            content_style: HollerBox.editor.tinymceCSS(),
          },
        }, (form_prompt) => {
          updateSetting({
            form_prompt,
          }, {
            overrides: {
              __state: 'form',
            },
          })
        })
      },
    },
    survey_button: {
      name: __('Start Button', 'holler-box'),
      render: ({
        survey_button_text = 'Start Survey',
        survey_button_color = '',
        survey_button_text_color = '',
        survey_button_align = 'center',
      }) => {
        return [
          singleControl({
            label: __('Button Text'),
            control: input({
              id: 'survey-button-text',
              value: survey_button_text,
            }),
          }),
          singleControl({
            label: __('Button Color'),
            control: input({
              id: 'survey-button-color',
              value: survey_button_color,
            }),
          }),
          singleControl({
            label: __('Text Color'),
            control: input({
              id: 'survey-button-text-color',
              value: survey_button_text_color,
            }),
          }),
          singleControl({
            label: __('Align'),
            control: select({
              id: 'survey-button-align',
              selected: survey_button_align,
              options: {
                left: __('Left'),
                center: __('Center'),
                right: __('Right'),
              },
            }),
          }),
        ].join('')
      },
      onMount: (settings, updateSetting) => {
        $('#survey-button-text').on('change input', e => {
          updateSetting({
            survey_button_text: e.target.value,
          })
        })

        $('#survey-button-align').on('change', e => {
          updateSetting({
            survey_button_align: $(e.target).val(),
          })
        })

        $('#survey-button-color').wpColorPicker({
          change: (e, ui) => {
            updateSetting({
              survey_button_color: ui.color.toString(),
            })
          },
        })

        $('#survey-button-text-color').wpColorPicker({
          change: (e, ui) => {
            updateSetting({
              survey_button_text_color: ui.color.toString(),
            })
          },
        })
      },
      css: ({
        id,
        survey_button_color = '',
        survey_button_text_color = '',
        survey_button_align = 'center',
      }) => {

        // language=CSS
        return `
            #${id} .survey-start .holler-box-button {
                background-color: ${survey_button_color};
                color: ${survey_button_text_color};
            }

            #${id} .survey-start {
                display: flex;
                justify-content: ${survey_button_align};
            }
        `
      },
    },
    embed: {
      name: __('Embed Code', 'holler-box'),
      render: ({
        embed_code = '',
      }) => {
        return [
          `<label>${__('Embed Code')}</label>`,
          textarea({
            id: 'embed-code',
            className: 'full-width code',
            value: embed_code,
          }),
          `<p class="holler-notice info">${__(
            'Shortcodes will be not rendered here.')}</p>`,
        ].join('')
      },
      onMount: (settings, updateSetting) => {
        $('#embed-code').on('input change', e => {
          updateSetting({
            embed_code: e.target.value,
          })
        })
      },
    },
    login_form: {
      name: __('Login Fields', 'holler-box'),
      render: ({
        username_placeholder = 'Username',
        password_placeholder = 'Password',
        enable_lost_password = true,
        enable_rememberme = true,
        rememberme_text = 'Remember Me',
        lost_password_text = 'Forgot your password?',
      }) => {
        return [
          `<label><b>${__('Username')}</b></label>`,
          singleControl({
            label: __('Placeholder'),
            control: input({
              id: 'username-placeholder',
              name: 'username_placeholder',
              value: username_placeholder,
            }),
          }),
          `<hr/>`,
          `<label><b>${__('Password')}</b></label>`,
          singleControl({
            label: __('Placeholder'),
            control: input({
              id: 'password-placeholder',
              name: 'password_placeholder',
              value: password_placeholder,
            }),
          }),
          `<hr/>`,
          `<label><b>${__('Remember Me')}</b></label>`,
          singleControl({
            label: __('Enable?'),
            control: toggle({
              id: 'enable-rememberme',
              name: 'enable_rememberme',
              checked: enable_rememberme,
            }),
          }),
          singleControl({
            label: __('Text'),
            control: input({
              id: 'rememberme-text',
              name: 'rememberme_text',
              value: rememberme_text,
            }),
          }),
          `<hr/>`,
          `<label><b>${__('Lost Password')}</b></label>`,
          singleControl({
            label: __('Enable?'),
            control: toggle({
              id: 'enable-lost-password',
              name: 'enable_lost_password',
              checked: enable_lost_password,
            }),
          }),
          singleControl({
            label: __('Text'),
            control: input({
              id: 'lost-pw-text',
              name: 'lost_password_text',
              value: lost_password_text,
            }),
          }),
        ].join('')
      },
      onMount: (settings, updateSetting) => {
        $('#username-placeholder, #password-placeholder, #lost-pw-text, #rememberme-text').on('input', e => {
          updateSetting({
            [e.target.name]: e.target.value,
          })
        })

        $('#enable-lost-password, #enable-rememberme').on('change', e => {
          updateSetting({
            [e.target.name]: e.target.checked,
          })
        })
      },
    },
    inverse_button: {
      name: __('Inverse Button', 'holler-box'),
      render: ({
        inverse_button_text = 'Create an account',
        inverse_button_link = '',
        enable_inverse_button = false,
      }) => {
        return [
          singleControl({
            label: __('Enable?'),
            control: toggle({
              id: 'enable-inverse-button',
              name: 'enable_inverse_button',
              checked: enable_inverse_button,
            }),
          }),
          singleControl({
            label: __('Button Text'),
            control: input({
              id: 'inverse-button-text',
              name: 'inverse_button_text',
              value: inverse_button_text,
            }),
          }),
          singleControl({
            label: __('Button Link'),
            control: input({
              type: 'url',
              id: 'inverse-button-link',
              name: 'inverse_button_link',
              value: inverse_button_link,
            }),
          }),
        ].join('')
      },
      onMount: (settings, updateSetting) => {
        $('#inverse-button-text, #inverse-button-link').on('input', e => {
          updateSetting({
            [e.target.name]: e.target.value,
          })
        })

        $('#enable-inverse-button').on('change', e => {
          updateSetting({
            [e.target.name]: e.target.checked,
          })
        })
      },
      css: ({
        id,
        button_color = '',
      }) => {

        // language=CSS
        return `
            #${id} .holler-box-button.inverse {
                background-color: transparent;
                color: ${button_color};
                border-color: ${button_color};
            }
        `
      },
    },
    after_login: {
      name: __('After Login', 'hollerbox-pro'),
      render: () => {
        return `<div id="m-after-login"></div>`
      },
      onMount: (settings, updateSetting) => {

        let {
          after_login = 'reload',
          redirect_to = '',
        } = settings

        const _morph = () => morph('.controls:has(#m-after-login)', render() )

        const render = () => Fragment([
          singleControlEl({
            label: __('After the user logs in'),
            control: Select({
              id: 'after_login',
              name: 'after_login',
              options: {
                reload: 'Reload the page',
                redirect: 'Redirect to a different page'
              },
              selected: after_login,
              onChange: e => {
                after_login = e.target.value
                updateSetting({
                  after_login,
                })
                _morph()
              },
            }),
          }),
          singleControlEl({
            label: __('Redirect to'),
            hidden: after_login !== 'redirect',
            control: Input({
              type: 'url',
              id: 'redirect_to',
              name: 'redirect_to',
              value: redirect_to,
              onChange: e => {
                redirect_to = e.target.value
                updateSetting({
                  redirect_to,
                })
              },
            }),
          }),
          `<div id="m-after-login"></div>`
        ])

        _morph()

      },
    },
  }

  const apiKeyInput = (key, label = __('API Key')) => {
    // language=HTML
    return `
		<label>${label}</label>
		${input({
			dataLpignore: true,
			id: 'crm-key',
			name: 'key',
			value: key,
			placeholder: 'api-key',
			type: 'password',
		})}
    `
  }

  const crmTags = (props = {}) => {

    let {
      label = __('Select which tags to add to the new contact record.'),
      id = 'crm-tags',
    } = props

    // language=HTML
    return `
		<div class="row">
			<div class="col">
				<p>${label}</p>
				<div id="${id}"></div>
			</div>
		</div>`
  }

  const crmTagPicker = ({
    selected,
    propName = 'tags',
    fetchTags,
    id = 'crm-tags',
    mapToItem = t => ({ id: t.id, text: t.name }),
    ...rest
  }) => {
    let timeout

    itemPicker(`#${id}`, {
      selected,
      placeholder: __('Type to add...'),
      tags: true,
      noneSelected: __('Select some tags...'),
      fetchOptions: (search, resolve) => {

        if (timeout) {
          clearTimeout(timeout)
        }

        timeout = setTimeout(() => {
          fetchTags(search).then(items => {
            resolve(items.map(mapToItem))
          }).catch(e => {
            return resolve([])
          })

        }, 1000)
      },
      ...rest,
    })
  }

  const crmConnectButton = () => {
    // language=HTML
    return `
		<p>
			<button class="holler-button secondary" id="crm-connect">
				${__('Connect')}
			</button>
		</p>`
  }

  const crmConnect = ({
    testConnection,
    updateIntegration,
  }) => {
    $('#crm-connect').on('click', e => {

      testConnection().then(success => {
        if (success) {
          updateIntegration({
            connected: true,
          }, true)
        } else {
          errorDialog({
            message: __('No tags found'),
          })
        }
      }).catch(e => {
        errorDialog({
          message: e.message,
        })
      })

    })
  }

  const ProIntegrations = {
    activecampaign: {
      id: 'activecampaign',
      group: 'crm',
      name: 'ActiveCampaign',
      //language=HTML
      icon: IntegrationIcons.activecampaign,
      edit: ({ tags, key = '', account = '', connected = false }) => {

        // language=HTML
        return `
			<div class="holler-rows-and-columns">
				<div class="row">
					<div class="col">
						<label>${__('Account Name')}</label>
						${input({
							dataLpignore: true,
							id: 'crm-account',
							name: 'account',
							value: account,
							placeholder: 'account-name',
						})}
					</div>
					<div class="col">
						${apiKeyInput(key)}
					</div>
				</div>
				${connected && key && account ? crmTags({
					id: 'crm-lists',
					label: __('Add the subscriber to these lists.'),
				}) : ''}
				${connected && key && account ? crmTags() : crmConnectButton()}
			</div>`
      },
      onMount: (integration, { updateIntegration, getState }) => {

        let {
          account,
          key,
          connected = false,
          tags = [],
          lists = [],
        } = integration

        const fetchTags = (search = '') => {
          return apiPost(HollerBox.routes.root + '/activecampaign', {
            search,
            key: getState().key,
            account: getState().account,
          }).then(r => r.tags)
        }

        const fetchLists = (search = '') => {
          return apiPost(HollerBox.routes.root + '/activecampaign/lists', {
            search,
            key: getState().key,
            account: getState().account,
          }).then(r => r.lists)
        }

        $('#crm-key, #crm-account').on('input', e => {
          let val = e.target.value
          updateIntegration({
            [e.target.name]: val,
          })
        })

        if (!key || !account || !connected) {

          crmConnect({
            testConnection: () => fetchTags().then(t => t.length),
            updateIntegration,
          })

          return
        }

        crmTagPicker({
          id: 'crm-lists',
          selected: lists,
          onChange: selected => updateIntegration({
            lists: selected,
          }),
          fetchTags: fetchLists,
          noneSelected: __('Select a list...'),
          tags: false,
        })

        crmTagPicker({
          selected: tags,
          mapToItem: f => ({ id: f.id, text: f.tag }),
          fetchTags,
          onChange: (selected) => updateIntegration({ tags: selected }),
        })
      },
    },
    convertkit: {
      id: 'convertkit',
      group: 'crm',
      name: 'ConvertKit',
      //language=HTML
      icon: IntegrationIcons.convertkit,
      edit: ({ forms, key = '', connected = false }) => {

        // language=HTML
        return `
			<div class="holler-rows-and-columns">
				<div class="row">
					<div class="col">
						${apiKeyInput(key)}
					</div>
				</div>
				${connected && key ? crmTags({
					label: __('Select which forms to add the subscriber to.'),
				}) : crmConnectButton()}
			</div>`
      },
      onMount: (integration, { getState, updateIntegration }) => {

        let {
          key,
          connected = false,
          forms = [],
        } = integration

        const fetchForms = (search = '') => {
          return apiPost(HollerBox.routes.root + '/convertkit', {
            search,
            key: getState().key,
          }).then(r => r.forms)
        }

        $('#crm-key').on('input', e => {
          let val = e.target.value
          updateIntegration({
            [e.target.name]: val,
          })
        })

        if (!key || !connected) {

          crmConnect({
            testConnection: () => fetchForms().then(f => f.length),
            updateIntegration,
          })

          return
        }

        crmTagPicker({
          selected: forms,
          onChange: selected => updateIntegration({
            forms: selected,
          }),
          fetchTags: fetchForms,
          noneSelected: __('Select a form...'),
          tags: false,
        })
      },
    },
    drip: {
      id: 'drip',
      group: 'crm',
      name: 'Drip',
      //language=HTML
      icon: IntegrationIcons.drip,
      edit: ({ tags, key = '', account = '', connected = false }) => {

        // language=HTML
        return `
			<div class="holler-rows-and-columns">
				<div class="row">
					<div class="col">
						<label>${__('Account ID')}</label>
						${input({
							dataLpignore: true,
							id: 'crm-account',
							name: 'account',
							value: account,
							placeholder: 'account-name',
						})}
					</div>
					<div class="col">
						${apiKeyInput(key)}
					</div>
				</div>
				${connected && key && account ? crmTags() : crmConnectButton()}
			</div>`
      },
      onMount: (integration, { updateIntegration, getState }) => {

        let {
          account,
          key,
          connected = false,
          tags = [],
        } = integration

        const fetchTags = (search = '') => {
          return apiPost(HollerBox.routes.root + '/drip', {
            search,
            key: getState().key,
            account: getState().account,
          }).then(r => r.tags)
        }

        $('#crm-key, #crm-account').on('input', e => {
          let val = e.target.value
          updateIntegration({
            [e.target.name]: val,
          })
        })

        if (!key || !account || !connected) {

          crmConnect({
            testConnection: () => fetchTags().then(t => t.length),
            updateIntegration,
          })

          return
        }

        crmTagPicker({
          selected: tags,
          mapToItem: t => ({ id: t, text: t }),
          fetchTags,
          onChange: (selected) => updateIntegration({ tags: selected }),
        })
      },
    },
    hubspot: {
      id: 'hubspot',
      group: 'crm',
      name: 'HubSpot',
      //language=HTML
      icon: IntegrationIcons.hubspot,
      edit: ({ key = '', connected = false }) => {

        // language=HTML
        return `
			<div class="holler-rows-and-columns">
				<div class="row">
					<div class="col">
						${apiKeyInput(key, __('Private App Access Token'))}
					</div>
				</div>
				${connected && key ? crmTags({
					id: 'crm-lists',
					label: __('Add the contact to these <b>Static</b> lists.'),
				}) : crmConnectButton()}
			</div>`
      },
      onMount: (integration, { getState, updateIntegration }) => {

        let {
          key,
          connected = false,
          lists = [],
        } = integration

        const fetchLists = (search = '') => {
          return apiPost(HollerBox.routes.root + '/hubspot', {
            search,
            key: getState().key,
          }).then(r => r.lists)
        }

        $('#crm-key').on('input', e => {
          let val = e.target.value
          updateIntegration({
            [e.target.name]: val,
          })
        })

        if (!key || !connected) {

          crmConnect({
            testConnection: () => fetchLists().then(l => l.length),
            updateIntegration,
          })

          return
        }

        crmTagPicker({
          id: 'crm-lists',
          selected: lists,
          onChange: selected => updateIntegration({
            lists: selected,
          }),
          fetchTags: fetchLists,
          mapToItem: l => ({ id: l.listId, text: l.name }),
          noneSelected: __('Select a list...'),
          tags: false,
        })
      },
    },
    keap: {
      id: 'keap',
      group: 'crm',
      name: 'Keap',
      //language=HTML
      icon: IntegrationIcons.keap,
    },
    mailchimp: {
      id: 'mailchimp',
      group: 'crm',
      name: 'MailChimp',
      //language=HTML
      icon: IntegrationIcons.mailchimp,

      edit: ({ key = '', connected = false, double_optin = false }) => {

        // language=HTML
        return `
			<div class="holler-rows-and-columns">
				<div class="row">
					<div class="col">
						${apiKeyInput(key)}
					</div>
				</div>
				${connected && key ? crmTags({
					id: 'crm-lists',
					label: __('Add the subscriber to these audiences.'),
				}) : crmConnectButton()}
				${connected && key ? crmTags({
					label: __(
						'Apply these tags. You <b>must</b> select one or more audiences first.'),
				}) : ''}
				${connected && key ? `<div class="display-flex gap-10">${__(
					'Require double opt-in?')} ${toggle({
					id: 'double-optin',
					checked: double_optin,
				})}</div>` : ''}
			</div>`
      },
      onMount: (integration, { getState, updateIntegration }) => {

        let {
          key,
          connected = false,
          lists = [],
          tags = [],
        } = integration

        const fetchLists = (search = '') => {
          return apiPost(HollerBox.routes.root + '/mailchimp/lists', {
            search,
            key: getState().key,
          }).then(r => r.lists)
        }

        const fetchTags = (search = '') => {
          return apiPost(HollerBox.routes.root + '/mailchimp/tags', {
            search,
            key: getState().key,
            lists: getState().lists.map(list => list.id),
          }).
          then(r => r.tags).
          then(tags => tags.length ? tags : ({ id: search, text: search }))
        }

        $('#crm-key').on('input', e => {
          let val = e.target.value
          updateIntegration({
            [e.target.name]: val,
          })
        })

        if (!key || !connected) {

          crmConnect({
            testConnection: () => fetchLists().then(l => l.length),
            updateIntegration,
          })

          return
        }

        $('#double-optin').on('change', e => {
          updateIntegration({
            double_optin: e.target.checked,
          })
        })

        crmTagPicker({
          id: 'crm-lists',
          selected: lists,
          onChange: selected => updateIntegration({
            lists: selected,
          }),
          fetchTags: fetchLists,
          noneSelected: __('Select a list...'),
          tags: false,
        })

        crmTagPicker({
          selected: tags,
          onChange: selected => updateIntegration({
            tags: selected,
          }),
          mapToItem: t => ({ id: t, text: t }),
          fetchTags,
          noneSelected: __('Select a tag...'),
        })
      },
    },
    mailpoet: {
      id: 'mailpoet',
      group: 'crm',
      name: 'MailPoet (v3)',
      //language=HTML
      icon: IntegrationIcons.mailpoet,
      beforeAdd: (add = () => {}) => {

        if (HollerBoxPro.installed.mailpoet) {
          return true
        }

        alert(
          __('MailPoet must be installed before you can add the integration.'))

        return false
      },
      edit: ({ key = '' }) => {

        if (!HollerBoxPro.installed.mailpoet) {
          return `<p class="holler-notice">
              ${__(
            'MailPoet must be installed before you can add the integration.')}
          </p>`
        }

        // language=HTML
        return `
			<div class="holler-rows-and-columns">
				${crmTags({
					id: 'crm-lists',
					label: __('Add the subscriber to these lists.'),
				})}
			</div>`
      },
      onMount: (integration, { getState, updateIntegration }) => {

        let {
          lists = [],
        } = integration

        const fetchLists = (search = '') => {
          return apiPost(HollerBox.routes.root + '/mailpoet', {
            search,
          }).then(r => r.lists)
        }

        crmTagPicker({
          id: 'crm-lists',
          selected: lists,
          onChange: selected => updateIntegration({
            lists: selected,
          }),
          fetchTags: fetchLists,
          noneSelected: __('Select a list...'),
          tags: false,
        })
      },
    },

  }

  // language=HTML
  const defaultYesNoContent = `
	  <h2>Want to join?</h2>
	  <p>Become part of our community for exclusive perks and benefits.</p>
  `

  let ProKeywords = {
    pro: __('Pro', 'hollerbox-pro'),
    banner: __('Banner', 'hollerbox-pro'),
    sidebar: __('Side Bar', 'hollerbox-pro'),
    survey: __('Survey', 'hollerbox-pro'),
    yesNo: __('Yes/No', 'hollerbox-pro'),
    embed: __('Embed', 'hollerbox-pro'),
    login: __('Login', 'hollerbox-pro'),
  }

  for (let keyword in ProKeywords) {
    Keywords[keyword] = ProKeywords[keyword]
  }

  const ProPopupTemplates = {
    popup_yes_no: {
      id: 'popup_yes_no',
      name: __('Popup with Yes/No'),
      keywords: [Keywords.pro, Keywords.popup, Keywords.yesNo, Keywords.button],
      controls: [
        Controls.position,
        Controls.overlay,
        Controls.close_button,
        Controls.content,
        ProControls.yesNoButton,
        ProControls.choseYes,
        ProControls.choseNo,
      ],
      defaults: {
        post_content: defaultYesNoContent,
        position: 'center-center',
        button_alignment: 'vertical',
        yes_text: 'Yes! Sign me up!',
        no_text: 'No thanks...',
      },
    },
    popup_yes_no_image_left: {
      id: 'popup_yes_no_image_left',
      name: __('Popup with Yes/No, Image Left'),
      keywords: [Keywords.pro, Keywords.popup, Keywords.yesNo, Keywords.button, Keywords.media, Keywords.image],
      controls: [
        Controls.position,
        Controls.overlay,
        Controls.close_button,
        Controls.image,
        Controls.content,
        ProControls.yesNoButton,
        ProControls.choseYes,
        ProControls.choseNo,
      ],
      defaults: {
        post_content: defaultYesNoContent,
        position: 'center-center',
        button_alignment: 'vertical',
        yes_text: 'Yes! Sign me up!',
        no_text: 'No thanks...',
        image_src: `${HollerBox.assets.root}/img/default/woman-yellow.png`,
      },
    },
    popup_yes_no_image_right: {
      id: 'popup_yes_no_image_right',
      name: __('Popup with Yes/No, Image Right'),
      keywords: [Keywords.pro, Keywords.popup, Keywords.yesNo, Keywords.button, Keywords.media, Keywords.image],
      controls: [
        Controls.position,
        Controls.overlay,
        Controls.close_button,
        Controls.image,
        Controls.content,
        ProControls.yesNoButton,
        ProControls.choseYes,
        ProControls.choseNo,
      ],
      defaults: {
        post_content: defaultYesNoContent,
        position: 'center-center',
        button_alignment: 'vertical',
        yes_text: 'Yes! Sign me up!',
        no_text: 'No thanks...',
        image_src: `${HollerBox.assets.root}/img/default/man-thumbs-up.png`,
      },
    },
    popup_with_login: {
      id: 'popup_with_login',
      name: __('Popup with Login Form'),
      keywords: [Keywords.pro, Keywords.popup, Keywords.login],
      controls: [
        Controls.position,
        Controls.overlay,
        Controls.close_button,
        Controls.content,
        ProControls.login_form,
        {
          ...Controls.button,
          name: __('Login Button'),
        },
        {
          ...ProControls.inverse_button,
          name: __('Create Account Button'),
        },
        ProControls.after_login
      ],
      defaults: {
        //language=HTML
        post_content: `
			<h1 style="text-align: center;"><span style="font-weight: 400;">🔒<strong> Access Denied</strong></span>
			</h1>
			<p style="text-align: center;">Please log in to access this content.</p>`,
        position: 'center-center',
        button_text: 'Login',
        enable_lost_password: true,
        enable_rememberme: true,
        disable_closing: true,
      },
    },
    popup_with_embed: {
      id: 'popup_with_embed',
      name: __('Popup with Embed'),
      keywords: [Keywords.pro, Keywords.popup, Keywords.media, Keywords.embed],
      controls: [
        Controls.position,
        Controls.overlay,
        Controls.close_button,
        Controls.content,
        ProControls.embed,
      ],
      defaults: {
        post_content: `<h1 style="text-align: center;"><span style="color: #000000;">WATCH THIS!</span></h1>
<p style="text-align: center;"><span style="color: #000000;">Watch how we can fix your problem below 👇...</span></p>`,
        position: 'center-center',
      },
    },
    popup_with_survey: {
      id: 'popup_with_survey',
      name: __('Popup with Multi-Step Survey'),
      keywords: [Keywords.pro, Keywords.popup, Keywords.survey],
      controls: [
        Controls.position,
        Controls.overlay,
        Controls.close_button,
        Controls.content,
        ProControls.survey_button,
        ProControls.survey,
        ProControls.form_prompt,
        Controls.fields,
        Controls.button,
        Controls.submit,
        Controls.integration,
      ],
      defaults: {
        post_content: defaultYesNoContent,
        position: 'center-center',
        button_alignment: 'vertical',
      },
    },
    popup_with_survey_links: {
      id: 'popup_with_survey_links',
      name: __('Popup with Survey Choices'),
      keywords: [Keywords.pro, Keywords.popup, Keywords.survey, Keywords.button],
      controls: [
        Controls.position,
        Controls.overlay,
        Controls.close_button,
        Controls.content,
        ProControls.survey_links,
      ],
      defaults: {
        post_content: defaultYesNoContent,
        position: 'center-center',
        button_alignment: 'vertical',
        choices: [
          { choice: 'Option A' },
          { choice: 'Option B' },
          { choice: 'Option C' },
        ],
      },
    },
    banner_fixed_image_right: {
      id: 'banner_fixed_image_right',
      name: __('Small Banner, Image Right'),
      keywords: [Keywords.pro, Keywords.banner, Keywords.image, Keywords.media],
      controls: [
        ProControls.position_banner,
        Controls.close_button,
        Controls.image,
        Controls.content,
      ],
      defaults: {
        post_content: '<p>Some cool content you should see!</p>',
        position: 'top',
        image_src: HollerBoxPro.assets.img.pdf,
      },
    },
    banner_fixed_image_left: {
      id: 'banner_fixed_image_left',
      name: __('Small Banner, Image Left'),
      keywords: [Keywords.pro, Keywords.banner, Keywords.image, Keywords.media],
      controls: [
        ProControls.position_banner,
        Controls.close_button,
        Controls.image,
        Controls.content,
      ],
      defaults: {
        post_content: '<p>Some cool content you should see!</p>',
        position: 'top',
        image_src: HollerBoxPro.assets.img.pdf,
      },
    },
    banner_standard: {
      id: 'banner_standard',
      name: __('Standard Banner'),
      keywords: [Keywords.pro, Keywords.banner, Keywords.basic],
      controls: [
        ProControls.position_banner,
        Controls.close_button,
        Controls.content,
      ],
      defaults: {
        post_content: '<p>Some cool content you should see!</p>',
        position: 'top',
      },
    },
    banner_image_right: {
      id: 'banner_image_right',
      name: __('Banner, Image Right'),
      keywords: [Keywords.pro, Keywords.banner, Keywords.image, Keywords.media],
      controls: [
        ProControls.position_banner,
        Controls.close_button,
        Controls.image,
        Controls.content,
      ],
      defaults: {
        post_content: '<p>Some cool content you should see!</p>',
        position: 'top',
        image_src: `${HollerBox.assets.root}/img/default/woman-shopping.png`,
      },
    },
    banner_image_left: {
      id: 'banner_image_left',
      name: __('Banner, Image Left'),
      keywords: [Keywords.pro, Keywords.banner, Keywords.image, Keywords.media],
      controls: [
        ProControls.position_banner,
        Controls.close_button,
        Controls.image,
        Controls.content,
      ],
      defaults: {
        post_content: '<p>Some cool content you should see!</p>',
        position: 'top',
        image_src: `${HollerBox.assets.root}/img/default/man-blue.png`,
      },
    },
    banner_with_button: {
      id: 'banner_with_button',
      name: __('Banner with Button'),
      keywords: [Keywords.pro, Keywords.banner, Keywords.button],
      controls: [
        ProControls.position_banner,
        Controls.close_button,
        Controls.content,
        Controls.link_button,
      ],
      defaults: {
        post_content: '<p>Some cool content you should see!</p>',
        position: 'top',
      },
    },
    banner_with_button_image_right: {
      id: 'banner_with_button_image_right',
      name: __('Banner with Button, Image Right'),
      keywords: [Keywords.pro, Keywords.banner, Keywords.button, Keywords.image, Keywords.media],
      controls: [
        ProControls.position_banner,
        Controls.close_button,
        Controls.image,
        Controls.content,
        Controls.link_button,
      ],
      defaults: {
        post_content: '<p>Some cool content you should see!</p>',
        position: 'top',
        image_src: `${HollerBox.assets.root}/img/default/woman-shopping.png`,
        button_text: 'Subscribe',
      },
    },
    banner_with_button_image_left: {
      id: 'banner_with_button_image_left',
      name: __('Banner with Button, Image Left'),
      keywords: [Keywords.pro, Keywords.banner, Keywords.button, Keywords.image, Keywords.media],
      controls: [
        ProControls.position_banner,
        Controls.close_button,
        Controls.image,
        Controls.content,
        Controls.link_button,
      ],
      defaults: {
        post_content: '<p>Some cool content you should see!</p>',
        position: 'top',
        image_src: `${HollerBox.assets.root}/img/default/woman-yellow.png`,
        button_text: 'Subscribe',
      },
    },
    banner_with_form: {
      id: 'banner_with_form',
      name: __('Banner with Form'),
      keywords: [Keywords.pro, Keywords.banner, Keywords.form, Keywords.email_only],
      controls: [
        ProControls.position_banner,
        Controls.close_button,
        Controls.content,
        Controls.fields_email_only,
        Controls.button,
        Controls.submit,
        Controls.integration,
      ],
      defaults: {
        post_content: '<p>Some cool content you should see!</p>',
        position: 'top',
      },
    },
    banner_with_form_image_right: {
      id: 'banner_with_form_image_right',
      name: __('Banner with Form, Image Right'),
      keywords: [Keywords.pro, Keywords.banner, Keywords.form, Keywords.email_only, Keywords.image],
      controls: [
        ProControls.position_banner,
        Controls.close_button,
        Controls.image,
        Controls.content,
        Controls.fields_email_only,
        Controls.button,
        Controls.submit,
        Controls.integration,
      ],
      defaults: {
        post_content: '<p>Some cool content you should see!</p>',
        position: 'top',
        image_src: `${HollerBox.assets.root}/img/default/man-thumbs-up.png`,
        button_text: 'Subscribe',
      },
    },
    banner_with_form_image_left: {
      id: 'banner_with_form_image_left',
      name: __('Banner with Form, Image Left'),
      keywords: [Keywords.pro, Keywords.banner, Keywords.form, Keywords.email_only, Keywords.image],
      controls: [
        ProControls.position_banner,
        Controls.close_button,
        Controls.image,
        Controls.content,
        Controls.fields_email_only,
        Controls.button,
        Controls.submit,
        Controls.integration,
      ],
      defaults: {
        post_content: '<p>Some cool content you should see!</p>',
        position: 'top',
        image_src: `${HollerBox.assets.root}/img/default/man-blue.png`,
        button_text: 'Subscribe',
      },
    },
    sidebar_standard: {
      id: 'sidebar_standard',
      name: __('Standard Sidebar'),
      keywords: [Keywords.pro, Keywords.sidebar, Keywords.basic],
      controls: [
        ProControls.position_sidebar,
        Controls.close_button,
        Controls.content,
      ],
      defaults: {
        post_content: '<p>Some cool content you should see!</p>',
        position: 'top',
      },
    },
    sidebar_with_form_below: {
      id: 'sidebar_with_form_below',
      name: __('Sidebar with Form Below'),
      keywords: [Keywords.pro, Keywords.sidebar, Keywords.form],
      controls: [
        ProControls.position_sidebar,
        Controls.close_button,
        Controls.content,
        Controls.fields,
        Controls.button,
        Controls.submit,
        Controls.integration,
      ],
      defaults: {
        post_content: '<p>Some cool content you should see!</p>',
        position: 'top',
        image_src: 'https://via.placeholder.com/600x200',
        button_text: 'Subscribe',
      },
    },
  }

  if (defaultFomoIntegration) {
    ProPopupTemplates.fomo = {
      id: 'fomo',
      name: __('FOMO (Sale Notification)'),
      keywords: [Keywords.pro, Keywords.notification],
      controls: [
        Controls.position,
        ProControls.fomo,
      ],
      defaults: {
        position: 'bottom-right',
        fomo_integration: defaultFomoIntegration,
        fomo_image: 'avatar',
        fomo_text: '{{name}} purchased {{product}}',
      },
    }
  }

  const ProAdvancedDisplayRules = {
    show_after_date: {
      name: __('Show after X date'),
      controls: ({ date = '' }) => {
        //language=HTML
        return input({ type: 'date', id: 'after-date', value: date })
      },
      onMount: (trigger, updateTrigger) => {
        $('#after-date').on('change', e => {
          updateTrigger({
            date: e.target.value,
          })
        })
      },
    },
    show_until_date: {
      name: __('Show until X date'),
      controls: ({ date = '' }) => {
        //language=HTML
        return input({ type: 'date', id: 'until-date', value: date })
      },
      onMount: (trigger, updateTrigger) => {
        $('#until-date').on('change', e => {
          updateTrigger({
            date: e.target.value,
          })
        })
      },
    },
    user_roles: {
      name: __('Show to X user roles'),
      controls: () => {
        //language=HTML
        return `
			<button class="holler-button secondary small sm"
			        id="select-user-roles">${__('Select roles')}
			</button>`
      },
      onMount: ({ selected = [] }, updateTrigger) => {

        let _selected = [...selected]

        $('#select-user-roles').on('click', e => {

          modal({
            width: 500,
            dialogClasses: 'overflow-visible has-header',
            // language=HTML
            content: `
				<div class="holler-header">
					<h3>${__('Select User Roles')}</h3>
					<button
						class="holler-button secondary text icon holler-modal-button-close"><span
						class="dashicons dashicons-no-alt"></span>
					</button>
				</div>
				<div id="holler-user-roles"></div>`,
            onOpen: () => {

              let options = [
                {
                  id: 'guest',
                  text: __('Guest', 'holler-box'),
                },
                ...HollerBoxPro.roles,
              ]

              itemPicker(`#holler-user-roles`, {
                selected: _selected,
                placeholder: __('Type to search...'),
                noneSelected: __('Select a role...'),
                onChange: (selected) => {
                  _selected = [...selected]
                  updateTrigger({ selected: _selected })
                },
                fetchOptions: (search, resolve) => {
                  let regex = new RegExp(search, 'i')
                  resolve(options.filter(({ text }) => text.match(regex)))
                },
              })
            },
          })
        })
      },
    },
  }

  if (HollerBoxPro.installed.pmp) {
    ProAdvancedDisplayRules.pmpro_level = {
      name: __('Show to PMPro members'),
      controls: ({ date = '' }) => {
        //language=HTML
        return `
			<button class="holler-button secondary small sm"
			        id="select-pmpro-levels">${__('Select levels')}
			</button>`
      },
      onMount: ({ selected = [] }, updateTrigger) => {

        let _selected = [...selected]

        $('#select-pmpro-levels').on('click', e => {

          modal({
            width: 500,
            dialogClasses: 'overflow-visible has-header',
            // language=HTML
            content: `
				<div class="holler-header">
					<h3>${__('Select PMPro Member Levels')}</h3>
					<button
						class="holler-button secondary text icon holler-modal-button-close"><span
						class="dashicons dashicons-no-alt"></span>
					</button>
				</div>
				<div id="holler-pmpro-levels"></div>`,
            onOpen: () => {

              let options = [
                { id: 'non_member', text: __('Non Members') },
                { id: 'any', text: __('Any Membership Level') },
                ...HollerBoxProPMP.map(level => ({
                  id: level.id,
                  text: level.name,
                })),
              ]

              itemPicker(`#holler-pmpro-levels`, {
                selected: _selected,
                placeholder: __('Type to search...'),
                noneSelected: __('Select a level...'),
                onChange: (selected) => {
                  _selected = [...selected]
                  updateTrigger({ selected: _selected })
                },
                fetchOptions: (search, resolve) => {
                  let regex = new RegExp(search, 'i')
                  resolve(options.filter(({ text }) => text.match(regex)))
                },
              })
            },
          })
        })
      },
    }
  }

  const ProTriggers = {
    after_inactivity: {
      name: __('Show After Inactivity'),
      controls: ({ time = '' }) => {
        //language=HTML
        return sprintf(__('Inactive for %s seconds'),
          input({ type: 'number', id: 'inactivity-time', value: time, min: 0 }))
      },
      onMount: (trigger, updateTrigger) => {
        $('#inactivity-time').on('change', e => {
          updateTrigger({
            time: e.target.value,
          })
        })
      },
    },
  }

  // Register the integrations with the Editor
  Object.keys(ProPopupTemplates).forEach(t => {
    Templates[t] = {
      id: t,
      ...ProPopupTemplates[t],
    }
  })

  Object.keys(ProIntegrations).forEach(i => {
    Integrations[i] = ProIntegrations[i]
  })

  Object.keys(ProAdvancedDisplayRules).forEach(r => {
    AdvancedDisplayRules[r] = ProAdvancedDisplayRules[r]
  })

  Object.keys(ProTriggers).forEach(r => {
    Triggers[r] = ProTriggers[r]
  })

})(jQuery)
