<?php

namespace GroundhoggZeroBounce;

use Groundhogg\Plugin;
use function Groundhogg\get_json_error;
use function Groundhogg\is_json_error;
use WP_Error;

class ZeroBounce {

	private $secret;
	private $base_url;

	/**
	 * @var ZeroBounce
	 */
	public static $instance;

	/**
	 * Get the instance of ZeroBounce SDK
	 *
	 * @return ZeroBounce
	 */
	public static function instance() {
		if ( self::$instance ) {
			return self::$instance;
		}

		self::$instance = new ZeroBounce();

		return self::$instance;
	}

	/**
	 * ZeroBounce constructor.
	 *
	 * Lazy load this bad boy
	 */
	public function __construct() {
		$this->init();
	}

	/**
	 * Get the API Key
	 *
	 * @return mixed
	 */
	public function get_api_key() {
		return $this->secret;
	}

	/**
	 * @param string $extra
	 *
	 * @return string
	 */
	public function get_url( $extra = '' ) {
		return $this->base_url . '/' . $extra;
	}

	/**
	 * init the options when ready
	 */
	public function init() {
		$this->secret   = get_option( 'gh_zerobounce_api_key' );
		$this->base_url = 'https://api.zerobounce.net/v2';
	}

	/**
	 * @return bool
	 */
	public function is_connected() {
		return ! empty( $this->secret ) && ! empty( $this->base_url );
	}

	/**
	 * @param string $endpoint the REST endpoint
	 * @param array  $body     the body of the request
	 * @param string $method   The request method
	 * @param array  $headers  optional headers to override a request
	 *
	 * @return object|WP_Error
	 */
	public function request( $endpoint, $body = [], $headers = [], $method = 'POST' ) {

		if ( ! $this->is_connected() ) {
			return new WP_Error( 'zb_no_api_key', 'No ZeroBounce API KEY has been set.' );
		}

		$body ['api_key'] = $this->secret;
		$method           = strtoupper( $method );
		$url              = sprintf( '%s/%s', $this->base_url, $endpoint );

		$args = [
			'method'  => $method,
			'headers' => $headers,
			'body'    => $body,
		];

		if ( $method === 'GET' ) {
			$response = wp_remote_get( $url, $args );
		} else {
			$response = wp_remote_post( $url, $args );
		}

		if ( ! $response ) {
			return new WP_Error( 'unknown_error', sprintf( 'Failed to initialize remote %s.', $method ) );
		}

		if ( is_wp_error( $response ) ) {
			return $response;
		}

		$json = json_decode( wp_remote_retrieve_body( $response ) );

		// Handle ZB errors
		if ( isset( $json->error ) ) {
			return new WP_Error( 'zb_error', $json->error );
		}

		return $json;
	}


	/**
	 * Validate email address using the ZeroBounce API.
	 *
	 * @param array $body -  needs email and ip_address param
	 * @param array $headers
	 *
	 * @return object|WP_Error
	 */
	public function verify_email( $body = [], $headers = [] ) {
		return $this->get( 'validate', $body, $headers );
	}

	/**
	 * Get number of remaining credits from ZeroBounce
	 *
	 * @return int|WP_Error
	 */
	public function get_credits() {
		$result = $this->get( 'getcredits', [], [] );

		if ( $result->Credits == '-1' ) {
			return new WP_Error( 'zb_error', 'Either your API key is invalid or your account has no credits.' );
		}

		return intval( $result->Credits );
	}

	/**
	 * GET Request Wrapper
	 *
	 * @param       $endpoint
	 * @param       $body
	 * @param array $headers
	 *
	 * @return object|WP_Error
	 */
	public function get( $endpoint, $body = [], $headers = [] ) {
		return $this->request( $endpoint, $body, $headers, 'GET' );
	}

	/**
	 * PUT Request Wrapper
	 *
	 * @param       $endpoint
	 * @param       $body
	 * @param array $headers
	 *
	 * @return object|WP_Error
	 */
	public function put( $endpoint, $body, $headers = [] ) {
		return $this->request( $endpoint, $body, $headers, 'PUT' );
	}

	/**
	 * POST Request Wrapper
	 *
	 * @param       $endpoint
	 * @param       $body
	 * @param array $headers
	 *
	 * @return object|WP_Error
	 */
	public function post( $endpoint, $body, $headers = [] ) {
		return $this->request( $endpoint, $body, $headers, 'POST' );
	}

	/**
	 * PATCH Request Wrapper
	 *
	 * @param       $endpoint
	 * @param       $body
	 * @param array $headers
	 *
	 * @return object|WP_Error
	 */
	public function patch( $endpoint, $body, $headers = [] ) {
		return $this->request( $endpoint, $body, $headers, 'PATCH' );
	}

	/**
	 * DELETE Request Wrapper
	 *
	 * @param       $endpoint
	 * @param       $body
	 * @param array $headers
	 *
	 * @return object|WP_Error
	 */
	public function delete( $endpoint, $body, $headers = [] ) {
		return $this->request( $endpoint, $body, $headers, 'DELETE' );
	}

}