<?php

namespace GroundhoggZeroBounce;

use Groundhogg\Bulk_Jobs\Bulk_Job;
use Groundhogg\Contact;
use Groundhogg\Contact_Query;
use Groundhogg\Preferences;
use function Groundhogg\admin_page_url;
use function Groundhogg\html;
use function Groundhogg\notices;

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

class Validate_Emails extends Bulk_Job {

	public function __construct() {

		add_action( 'groundhogg/tools/misc', [ $this, 'tools_page' ] );

		parent::__construct();
	}

	/**
	 * Show the Misc ZeroBounce tool
	 */
	public function tools_page() {
		?>
        <div class="gh-panel tool">
        <div class="gh-panel-header">
            <h2 class="hndle"><?php _e( 'ZeroBounce', 'groundhogg' ); ?></h2>
        </div>
        <div class="inside">
            <p><?php _e( 'Force validate your contacts through the ZeroBounce validation API.', 'groundhogg' ); ?></p>
            <p class="gh-input-group">
				<?php

				echo html()->e( 'a', [
					'class' => 'gh-button secondary',
					'href'  => $this->get_start_url(),
				], __( 'Force Validate', 'groundhogg' ) );

				echo html()->e( 'a', [
					'class' => 'gh-button secondary',
					'href'  => admin_page_url( 'gh_tools', [
						'tab'    => 'misc',
						'action' => 'zb_report'
					] ),
				], __( 'View Report', 'groundhogg' ) ); ?>
            </p>
        </div>
        </div><?php
	}

	/**
	 * Get the action reference.
	 *
	 * @return string
	 */
	function get_action() {
		return 'zb_validate_emails';
	}

	/**
	 * Get an array of items someway somehow
	 *
	 * @param $items array
	 *
	 * @return array
	 */
	public function query( $items ) {

		if ( ! current_user_can( 'edit_contacts' ) ) {
			return $items;
		}

		$query = new Contact_Query( [
			'optin_status' => [
				Preferences::UNCONFIRMED,
			]
		] );

		$total = $query->count();

		return range( 0, ceil( $total / self::BATCH_SIZE ) );
	}

	const BATCH_SIZE = 30;

	/**
	 * Get the maximum number of items which can be processed at a time.
	 *
	 * @param $max   int
	 * @param $items array
	 *
	 * @return int
	 */
	public function max_items( $max, $items ) {
		return 1;
	}

	/**
	 * Process an item
	 *
	 * @param $item mixed
	 *
	 * @return void
	 */
	protected function process_item( $page ) {
		if ( ! current_user_can( 'edit_contacts' ) ) {
			return;
		}

		$page = absint( $page );

		$query = new Contact_Query( [
			'limit'        => self::BATCH_SIZE,
			'offset'       => $page * self::BATCH_SIZE,
			'optin_status' => [
				Preferences::UNCONFIRMED,
			]
		] );

		$contacts = $query->query( [], true );

		if ( empty( $contacts ) ) {
			return;
		}

		$requests = array_map( function ( $contact ) {
			return [
				'type'    => \Requests::GET,
				'url'     => ZeroBounce::instance()->get_url( 'validate' ),
				'data'    => [
					'api_key'    => ZeroBounce::instance()->get_api_key(),
					'email'      => $contact->get_email(),
					'ip_address' => $contact->get_ip_address()
				],
				'options' => [
					'complete' => [ $this, 'handle_response' ]
				]
			];
		}, $contacts );

		\Requests::request_multiple( $requests );
	}

	/**
	 * Handle the ZeroBounce response
	 *
	 * @param $response \Requests_Response|\Requests_Exception
	 */
	public function handle_response( $response ) {

		// Exit if an exception
		if ( is_a( $response, \Requests_Exception::class ) ) {
			return;
		}

		$json = json_decode( $response->body );

		if ( $response->status_code != 200 ) {
			notices()->add( 'error', $json->error, 'error' );

			return;
		}

		$email = $json->address;

		$contact = new Contact( $email );

		$contact->update_meta( '_zb_status', $json->status );
	}

	/**
	 * Do stuff before the loop
	 *
	 * @return void
	 */
	protected function pre_loop() {
	}

	/**
	 * do stuff after the loop
	 *
	 * @return void
	 */
	protected function post_loop() {
	}

	/**
	 * Cleanup any options/transients/notices after the bulk job has been processed.
	 *
	 * @return void
	 */
	protected function clean_up() {
	}

	/**
	 * Get the return URL
	 *
	 * @return string
	 */
	protected function get_return_url() {
		return admin_page_url( 'gh_tools', [
			'tab'    => 'misc',
			'action' => 'zb_report'
		] );
	}
}
