<?php

namespace GroundhoggZeroBounce;

use Groundhogg\Contact;
use Groundhogg\Extension;
use Groundhogg\Preferences;
use function Groundhogg\action_url;
use function Groundhogg\array_map_with_keys;
use function Groundhogg\get_url_var;
use function Groundhogg\html;
use function Groundhogg\notices;

class Plugin extends Extension {


	/**
	 * Override the parent instance.
	 *
	 * @var Plugin
	 */
	public static $instance;

	/**
	 * The ZB sdk
	 *
	 * @var ZeroBounce
	 */
	public $zerobounce;

	/**
	 * Include any files.
	 *
	 * @return void
	 */
	public function includes() {
		require __DIR__ . '/functions.php';
	}

	/**
	 * Init any components that need to be added.
	 *
	 * @return void
	 */
	public function init_components() {
		$this->zerobounce = new ZeroBounce();

		add_action( 'admin_init', [ $this, 'handle_test' ] );
	}

	/**
	 * Get the ID number for the download in EDD Store
	 *
	 * @return int
	 */
	public function get_download_id() {
		return 38376;
	}

	/**
	 * Get the version #
	 *
	 * @return mixed
	 */
	public function get_version() {
		return GROUNDHOGG_ZEROBOUNCE_VERSION;
	}

	/**
	 * Register the ZeroBounce settings
	 *
	 * @param array[] $settings
	 *
	 * @return array[]|array[]
	 */
	public function register_settings( $settings ) {

		$settings['gh_zerobounce_api_key'] = [
			'id'      => 'gh_zerobounce_api_key',
			'section' => 'zerobounce-setting',
			'label'   => __( 'API key', 'groundhogg-zerobounce' ),
			'desc'    => __( 'Enter your ZeroBounce API key from your account. Don\'t have an API key? <b><a href="https://groundho.gg/zerobounce" target="_blank">Get one now!</a></b>', 'groundhogg-zerobounce' ),
			'type'    => 'input',
			'atts'    => [
				'type' => 'password',
				'name' => 'gh_zerobounce_api_key',
				'id'   => 'gh_zerobounce_api_key',
			]
		];

		$settings['gh_zb_sync_invalid_statuses'] = [
			'id'      => 'gh_zb_sync_invalid_statuses',
			'section' => 'zerobounce-setting',
			'label'   => __( 'Which email statuses should impact marketability?', 'groundhogg-zerobounce' ),
			'desc'    => __( 'For the selected statuses the marketability of a contact will be set to <b>unmarketable</b>. ', 'groundhogg-zerobounce' ),
			'type'    => 'select2',
			'atts'    => [
				'multiple' => true,
				'data'     => array_map_with_keys( array_combine( get_zb_invalid_statuses(), get_zb_invalid_statuses() ), function ( $status ) {
					return get_zb_status_label( $status );
				} ),
				'name'     => 'gh_zb_sync_invalid_statuses[]',
				'id'       => 'gh_zb_sync_invalid_statuses',
			]
		];

		$settings['gh_zb_validate_free_inboxes'] = [
			'id'      => 'gh_zb_validate_free_inboxes',
			'section' => 'zerobounce-setting',
			'label'   => __( 'Validate public/free email addresses?', 'groundhogg-zerobounce' ),
			'desc'    => __( 'By default free/public emails like <b>@gmail.com</b> and <b>@hotmail.com</b> are not validated. When this setting is disabled all free public inbox providers will return a <code>valid</code> status automatically.', 'groundhogg-zerobounce' ),
			'type'    => 'checkbox',
			'atts'    => array(
				'label' => __( 'Enable' ),
				'name'  => 'gh_zb_validate_free_inboxes',
				'id'    => 'gh_zb_validate_free_inboxes',
				'value' => 'on',
			),
		];

		return $settings;
	}

	/**
	 * Add ZB Setting
	 *
	 * @param array[] $sections
	 *
	 * @return array[]|array[]
	 */
	public function register_settings_sections( $sections ) {
		$sections['zerobounce-setting'] = [
			'id'       => 'zerobounce-setting',
			'title'    => _x( 'ZeroBounce Settings', 'settings_sections', 'groundhogg-zerobounce' ),
			'tab'      => 'integrations',
			'callback' => [ $this, 'test_zero_bounce' ],
		];

		return $sections;
	}

	public function test_zero_bounce() {

		echo html()->e( 'a', [
			'class' => 'button button-secondary',
			'href'  => action_url( 'test_zerobounce' ),
		], __( 'Test Connection' ) );

	}

	public function handle_test() {
		if ( get_url_var( 'action' ) !== 'test_zerobounce' || ! wp_verify_nonce( get_url_var( '_wpnonce' ), 'test_zerobounce' ) ) {
			return;
		}

		$result = ZeroBounce::instance()->get_credits();

		if ( is_wp_error( $result ) ) {
			notices()->add( $result );
		} else {
			notices()->add( 'success', __( 'ZeroBounce is connected!' ) );
		}
	}

	public function register_bulk_jobs( $manager ) {
		$manager->zerobounce = new Validate_Emails();
	}

	/**
	 * @return string
	 */
	public function get_plugin_file() {
		return GROUNDHOGG_ZEROBOUNCE__FILE__;
	}

	/**
	 * Register autoloader.
	 *
	 * Groundhogg autoloader loads all the classes needed to run the plugin.
	 *
	 * @since  1.6.0
	 * @access private
	 */
	protected function register_autoloader() {
		require GROUNDHOGG_ZEROBOUNCE_PATH . 'includes/autoloader.php';
		Autoloader::run();
	}
}

Plugin::instance();
