<?php

namespace GroundhoggZeroBounce;

use Groundhogg\Contact;
use Groundhogg\Contact_Query;
use function Groundhogg\admin_page_url;
use function Groundhogg\get_array_var;
use function Groundhogg\html;
use function Groundhogg\is_free_email_provider;
use function Groundhogg\is_option_enabled;
use function Groundhogg\is_sending;
use function Groundhogg\notices;

/**
 * List of invalid email status
 *
 * @return array
 */
function get_zb_invalid_statuses() {
	return [
		'invalid',
		'spamtrap',
		'abuse',
		'do_not_mail'
	];
}

/**
 * Check if a given status is invalid per the settings
 *
 * @param $status
 *
 * @return bool
 */
function is_zb_invalid_status( $status ) {
	$statuses = get_option( 'gh_zb_sync_invalid_statuses', get_zb_invalid_statuses() );

	if ( ! is_array( $statuses ) ) {
		$statuses = get_zb_invalid_statuses();
	}

	return in_array( $status, $statuses );
}

/**
 * Get array of statuses and labels
 *
 * @return array
 */
function get_zb_statuses() {

	$statuses = [
		'valid'       => __( 'Valid' ),
		'catch-all'   => __( 'Catch-All' ),
		'invalid'     => __( 'Invalid' ),
		'spamtrap'    => __( 'Spam Trap' ),
		'abuse'       => __( 'Abuse' ),
		'do_not_mail' => __( 'Do Not Mail' ),
		'unknown'     => __( 'Unknown' ),
	];

	return $statuses;
}

function is_zb_status( $maybe_status ){
	return in_array( $maybe_status, array_keys( get_zb_statuses() ) );
}

/**
 * Get the label for a certain status
 *
 * @param $status string
 *
 * @return bool|mixed
 */
function get_zb_status_label( $status ) {
	return get_array_var( get_zb_statuses(), $status, '' );
}

/**
 * Output the ZB status pill
 *
 * @param $status
 */
function zb_status_pill( $status ) {
	?>
    <a href="<?php echo admin_page_url( 'gh_contacts', [
		'meta_key'   => '_zb_status',
		'meta_value' => $status
	] ) ?>"><span
                class="pill sm <?php echo in_array( $status, get_zb_invalid_statuses() ) ? 'red' : 'green' ?>"><?php echo get_zb_status_label( $status ) ?></span></a>
	<?php
}

/**
 * Fetch the ZeroBounce status of a contact
 *
 * @param Contact $contact
 *
 * @return false|string false if unable to get status from ZeroBounce, string status if success
 */
function fetch_zb_status( $contact ) {

	// Error lock is in place
	if ( get_transient( 'gh_zb_lock' ) === '1' ) {
		return false;
	}

	if ( ! is_option_enabled( 'gh_zb_validate_free_inboxes' ) && is_free_email_provider( $contact->get_email() ) ){
		return 'valid';
	}

	$response = ZeroBounce::instance()->verify_email( [
		'email'      => $contact->get_email(),
		'ip_address' => $contact->get_ip_address()
	] );

	if ( is_wp_error( $response ) ) {

		// Set a temporary error lock to avoid making unnecessary requests
		set_transient( 'gh_zb_lock', '1', MINUTE_IN_SECONDS );

		// Most likely a credits or api key error
		if ( $response->get_error_code() === 'zb_error' ) {
			notices()->add( 'not_enough_zb_credits', __( 'You are out of ZeroBounce credits! <a href="https://groundho.gg/zerobounce" target="_blank">Purchase more credits now!</a>', 'groundhogg-zerobounce' ), 'warning' );
		}

		return false;
	}

	$status = $response->status;

	if ( ! in_array( $status, array_keys( get_zb_statuses() ) ) ) {
		return false;
	}

	$contact->update_meta( '_zb_status', $status );

	return $status;
}

add_filter( 'groundhogg/contact/is_marketable', __NAMESPACE__ . '\filter_marketability', 10, 2 );

/**
 * If the ZB status is invalid, filter marketablilty
 *
 * @param $is_marketable bool
 * @param $contact       Contact
 */
function filter_marketability( $is_marketable, $contact ) {

	// If the contact is already not marketable no use in wasting a credit
	if ( ! $is_marketable ) {
		return $is_marketable;
	}

	// Assume email address must be valid for confirmed contacts
	if ( $contact->is_confirmed() ) {
		return true;
	}

	/**
	 * Filter the status, allows returning a status without fetching from ZeroBounce
	 *
	 * if $status is true, exit with the original marketability status
	 * if $status is false, the status will be fetched from the api
	 * Otherwise, the status will be compared against valid statuses
	 *
	 * @param string $status the zb_status
	 * @param Contact $contact
	 */
	$status = apply_filters( 'groundhogg/zerobounce/zb_status', $contact->get_meta( '_zb_status' ), $contact );

	if ( $status === true ){
		return $is_marketable;
	}

	// If the status is not recognized, fetch from the API
	if ( ! is_zb_status( $status ) ) {
		$status = fetch_zb_status( $contact );
	}

	// Re-check, because fetch_zb_status is not guaranteed to return a valid status
	if ( ! is_zb_status( $status ) ){
		return $is_marketable;
	}

	// If ZB status is not in invalid statuses, return true
	return ! is_zb_invalid_status( $status );
}

add_filter( 'groundhogg/preferences/optin_status_text', __NAMESPACE__ . '\filter_optin_status_text', 10, 2 );

/**
 * Filter the optin status text to display a relevant message about ZeroBounce
 *
 * @param $text
 * @param $contact
 */
function filter_optin_status_text( $text, $contact ) {

	// Get the status
	$status = $contact->get_meta( '_zb_status' );

	if ( ! $status ) {
		return $text;
	}

	if ( in_array( $status, get_zb_invalid_statuses() ) ) {
		return sprintf( __( 'ZeroBounce marked this email as <b>%s</b>', 'groundhogg-zerobounce' ), get_zb_status_label( $status ) );
	}

	return $text;
}


add_action( 'groundhogg/admin/gh_tools/display/misc_zb_report', __NAMESPACE__ . '\show_bulk_validate_report' );

/**
 * Show a tab report that users can look at to see the number of statuses for each ZB status, and from there decide what to do with them.
 */
function show_bulk_validate_report() {

	?><h2><?php _e( 'ZeroBounce Report', 'groundhogg' ) ?></h2><?php

	$query = new Contact_Query();

	$rows = [];

	foreach ( get_zb_statuses() as $status => $label ) {
		$rows[] = [
			$label,
			html()->e( 'a', [
				'href' => admin_page_url( 'gh_contacts', [
					'meta_key'   => '_zb_status',
					'meta_value' => $status,
				] )
			], number_format_i18n( $query->count( [
				'meta_key'   => '_zb_status',
				'meta_value' => $status,
			] ) ), false )
		];
	}

	html()->list_table( [], [
		__( 'Status' ),
		__( 'Contacts' ),
	], $rows );

}

add_action( 'groundhogg/contact/record/contact_info/after', __NAMESPACE__ . '\zb_status_dropdown', 5 );

/**
 * Show a dropdown to allow the admins to override the ZB status
 *
 * @param $contact Contact
 */
function zb_status_dropdown( $contact ) {
	?>
    <div class="gh-rows-and-columns" style="margin-top: 10px">
        <div class="gh-row">
            <div class="gh-col">
                <label for="_zb_status"><?php _e( 'ZeroBounce Status' ) ?></label>
				<?php echo html()->dropdown( [
					'name'     => '_zb_status',
					'id'       => '_zb_status',
					'options'  => get_zb_statuses(),
					'selected' => $contact->get_meta( '_zb_status' )
				] ); ?>
            </div>
        </div>
    </div>
	<?php
}

/**
 * Allow _zb_status to be saved
 */
add_filter( 'groundhogg/contact/update/basic_fields', function ( $fields ) {
	$fields[] = '_zb_status';

	return $fields;
} );

/**
 * Add ZB mappable fields
 *
 * @param $fields
 *
 * @return mixed
 */
function add_zb_mappable_fields( $fields ){

	$fields[ 'ZeroBounce' ] = [
		'_zb_status' => 'ZB Status'
	];

	return $fields;
}

add_filter( 'groundhogg/mappable_fields', __NAMESPACE__ . '\add_zb_mappable_fields' );

/**
 * Map the _zb_status field accordingly
 *
 * @param $field
 * @param $value
 * @param $args
 * @param $meta
 */
function map_zb_status( $field, $value, &$args, &$meta ){
	if ( $field === '_zb_status' && is_zb_status( $value ) ){
		$meta['_zb_status'] = $value;
	}
}

add_action( 'groundhogg/update_contact_with_map/default', __NAMESPACE__ . '\map_zb_status', 10, 4 );
