<?php

namespace GroundhoggReplacements;

use function Groundhogg\do_replacements;
use Groundhogg\Replacements;
use \WP_Error;
use Groundhogg\Plugin;
use function Groundhogg\is_option_enabled;

class Custom_Replacements {

	/**
	 * @var array
	 */
	private $codes;

	public function __construct() {
		$this->codes = Plugin::$instance->settings->get_option( 'gh_custom_replacements' );

		if ( ! $this->codes ) {
			return;
		}

		add_action( 'groundhogg/replacements/init', [ $this, 'setup_codes' ] );
	}

	/**
	 * Adds P tags to the content...
	 *
	 * @param $content
	 *
	 * @return string
	 */
	public function maybe_autop_content( $content ) {
		// if there are multiple lines we can autop, but otherwise let's not.
		return count( array_filter( explode( PHP_EOL, $content ) ) ) > 1 ? wpautop( $content ) : $content;
	}

	/**
	 * @return array
	 */
	public function get_codes() {
		if ( ! $this->codes ) {
			return [];
		}

		return $this->codes;
	}

	/**
	 * @param $code string
	 *
	 * @return array|mixed
	 */
	public function get_replacement( $code ) {
		if ( ! $this->codes ) {
			$this->codes = [];
		}

		if ( ! key_exists( $code, $this->codes ) ) {
			return false;
		}

		return $this->codes[ $code ];
	}

	/**
	 * Add a new replacement code
	 *
	 * @param $code        string
	 * @param $content     string
	 * @param $description string
	 * @param string $name
	 *
	 * @return bool|WP_Error
	 */
	public function add( $code, $content = '', $description = '', $name = '' ) {

		if ( ! $this->codes ) {
			$this->codes = [];
		}

		if ( key_exists( $code, $this->codes ) ) {
			return new WP_Error( 'code_exists', _x( 'This code already exists.', 'notice', 'groundhogg' ) );
		}

		$this->codes[ $code ] = [
			'code'        => $code,
			'content'     => $content,
			'description' => $description,
			'name'        => $name
		];

		return \Groundhogg\Plugin::$instance->settings->update_option( 'gh_custom_replacements', $this->codes );
	}

	/**
	 * Update an existing replacement code
	 *
	 * @param $code        string
	 * @param $content     string
	 * @param $description string
	 *
	 * @return bool
	 */
	public function update( $code, $content, $description, $name ) {
		if ( ! $this->codes ) {
			$this->codes = [];
		}

		if ( ! key_exists( $code, $this->codes ) ) {
			return $this->add( $code, $content, $description );
		}

		$this->codes[ $code ] = [
			'code'        => $code,
			'content'     => $content,
			'name'        => $name,
			'description' => $description,
		];

		return \Groundhogg\Plugin::$instance->settings->update_option( 'gh_custom_replacements', $this->codes );
	}

	/**
	 * Delete an existing replacement code
	 *
	 * @param $code
	 *
	 * @return bool
	 */
	public function delete( $code ) {
		unset( $this->codes[ $code ] );

		return \Groundhogg\Plugin::$instance->settings->update_option( 'gh_custom_replacements', $this->codes );
	}

	/**
	 * @param $replacements Replacements
	 */
	public function setup_codes( $replacements ) {
		if ( empty( $this->codes ) ) {
			return;
		}

		$replacements->add_group( 'custom_replacements', __( 'Custom Replacements', 'groundhogg-replacements' ) );

		$this->codes = apply_filters( 'groundhogg/custom_replacements/defaults', $this->codes );

		foreach ( $this->codes as $code => $args ) {

			$args = wp_parse_args( $args, [
				'code'        => '',
				'description' => '',
				'name'        => '',
				'group'       => 'custom_replacements',
			] );

			extract( $args );

			/**
			 * @var $code string
			 * @var $description string
			 * @var $name string
			 * @var $group string
			 */

			$replacements->add( $code, [
				$this,
				'replacement'
			], $description, $name, $group );
		}
	}

	/**
	 * Get the content of a custom replacement
	 *
	 * @param $code
	 *
	 * @return string
	 */
	private function get_replacement_content( $code ) {
		if ( ! key_exists( $code, $this->codes ) ) {
			return false;
		}

		return $this->codes[ $code ]['content'];
	}

	/**
	 * Process the custom replacement
	 *
	 * @param $contact_id int
	 * @param $code       string
	 *
	 * @return string
	 */
	public function replacement( $contact_id, $code ) {
		$content = $this->get_replacement_content( $code );
		$content = $this->maybe_autop_content( do_shortcode( do_replacements( $content, $contact_id ) ) );

		return apply_filters( 'groundhogg/custom_replacements/content', $content );
	}
}