<?php

namespace GroundhoggReplacements;

use \WP_List_Table;
use function Groundhogg\get_array_var;

/**
 * Replacements Table Class
 *
 * This class shows the data table for accessing information about a replacement code.
 *
 * @since       0.1
 * @subpackage  Modules/Replacements
 * @copyright   Copyright (c) 2018, Adrian Tobey
 * @license     http://opensource.org/licenses/gpl-2.0.php GNU Public License
 * @package     groundhogg
 */

// Exit if accessed directly
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

// WP_List_Table is not loaded automatically so we need to load it in our application
if ( ! class_exists( 'WP_List_Table' ) ) {
	require_once( ABSPATH . 'wp-admin/includes/class-wp-list-table.php' );
}

class Replacements_Table extends WP_List_Table {

	/**
	 * @var array
	 */
	public $codes;

	/**
	 * TT_Example_List_Table constructor.
	 *
	 * REQUIRED. Set up a constructor that references the parent constructor. We
	 * use the parent reference to set some default configs.
	 */
	public function __construct() {
		// Set parent defaults.
		$this->codes = Plugin::$instance->custom_replacements->get_codes();

		parent::__construct( array(
			'singular' => 'replacement',     // Singular name of the listed records.
			'plural'   => 'replacements',    // Plural name of the listed records.
			'ajax'     => false,       // Does this table support ajax?
		) );
	}

	/**
	 * @return array An associative array containing column information.
	 * @see WP_List_Table::::single_row_columns()
	 */
	public function get_columns() {
		$columns = array(
			'cb'          => '<input type="checkbox" />', // Render a checkbox instead of text.
			'name'        => _x( 'Name', 'Column label', 'groundhogg' ),
			'code'        => _x( 'Code', 'Column label', 'groundhogg' ),
			'description' => _x( 'Description', 'Column label', 'groundhogg' ),
//			'content'     => _x( 'Content', 'Column label', 'groundhogg' ),
		);

		return $columns;
	}

	/**
	 * @return array An associative array containing all the columns that should be sortable.
	 */
	protected function get_sortable_columns() {
		$sortable_columns = array(
			'code' => array( 'code', false ),
		);

		return $sortable_columns;
	}

	/**
	 * @param $replacement array
	 *
	 * @return string
	 */
	protected function column_name( $replacement ) {
		return get_array_var( $replacement, 'name', $replacement['code'] );
	}

	/**
	 * @param $replacement array
	 *
	 * @return string
	 */
	protected function column_code( $replacement ) {
		return "<input type=\"text\" onfocus=\"this.select()\" class=\"input code\" value=\"{{$replacement['code']}}\" readonly/>";
	}

	/**
	 * @param $replacement array
	 *
	 * @return mixed
	 */
	protected function column_description( $replacement ) {
		return $replacement['description'];
	}

	/**
	 * @param $replacement array
	 *
	 * @return mixed
	 */
	protected function column_content( $replacement ) {
		return esc_html( apply_filters( 'the_content', $replacement['content'] ) );
	}

	/**
	 * Get default column value.
	 *
	 * @param object $replacement A singular item (one full row's worth of data).
	 * @param string $column_name The name/slug of the column to be processed.
	 *
	 * @return string Text or HTML to be placed inside the column <td>.
	 */
	protected function column_default( $replacement, $column_name ) {

		return print_r( $replacement[ $column_name ], true );

	}

	/**
	 * @param object $replacement A singular item (one full row's worth of data).
	 *
	 * @return string Text to be placed inside the column <td>.
	 */
	protected function column_cb( $replacement ) {
		return sprintf(
			'<input type="checkbox" name="%1$s[]" value="%2$s" />',
			$this->_args['singular'],  // Let's simply repurpose the table's singular label ("movie").
			$replacement['code']              // The value of the checkbox should be the record's ID.
		);
	}

	/**
	 * Generates and displays row action replacement.
	 *
	 * @param object $replacement array        Contact being acted upon.
	 * @param string $column_name Current column name.
	 * @param string $primary Primary column name.
	 *
	 * @return string Row elements output for posts.
	 */
	protected function handle_row_actions( $replacement, $column_name, $primary ) {
		if ( $primary !== $column_name ) {
			return '';
		}

		$actions = array();
		$title   = $replacement['code'];

		$actions['edit'] = sprintf(
			'<a href="%s" class="edit" aria-label="%s">%s</a>',
			/* translators: %s: title */
			admin_url( 'admin.php?page=gh_replacements&action=edit&replacement=' . $replacement['code'] ),
			esc_attr( sprintf( __( 'Edit' ), $title ) ),
			__( 'Edit' )
		);

		$actions['delete'] = sprintf(
			'<a href="%s" class="submitdelete" aria-label="%s">%s</a>',
			wp_nonce_url( admin_url( 'admin.php?page=gh_replacements&replacement=' . $replacement['code'] . '&action=delete' ) ),
			/* translators: %s: title */
			esc_attr( sprintf( __( 'Delete &#8220;%s&#8221; Permanently' ), $title ) ),
			__( 'Delete' )
		);

		return $this->row_actions( $actions );
	}

	/**
	 * @return array An associative array containing all the bulk elements.
	 */
	protected function get_bulk_actions() {
		$actions = array(
			'delete' => _x( 'Delete', 'List table bulk action', 'groundhogg' ),
		);

		return apply_filters( 'wpgh_replacement_bulk_actions', $actions );
	}

	/**
	 * Prepares the list of items for displaying.
	 * @global wpdb $wpdb
	 * @uses $this->_column_headers
	 * @uses $this->items
	 * @uses $this->get_columns()
	 * @uses $this->get_sortable_columns()
	 * @uses $this->get_pagenum()
	 * @uses $this->set_pagination_args()
	 */
	function prepare_items() {

		$per_page = 30;

		$columns  = $this->get_columns();
		$hidden   = array();
		$sortable = $this->get_sortable_columns();

		$this->_column_headers = array( $columns, $hidden, $sortable );
		$data                  = [];
		if ( isset( $_REQUEST['s'] ) ) {
			$search = sanitize_textarea_field( $_REQUEST['s'] );
			foreach ( $this->codes as $code => $args ) {
				foreach ( $args as $i => $arg ) {
					if ( strpos( $arg, $search ) !== false ) {
						$data[ $code ] = $args;
						continue;
					}
				}
			}

		} else {
			$data = $this->codes;
		}


		/*
		 * Sort the data
		 */
		usort( $data, array( $this, 'usort_reorder' ) );

		$current_page = $this->get_pagenum();

		$total_items = count( $data );

		$data = array_slice( $data, ( ( $current_page - 1 ) * $per_page ), $per_page );

		$this->items = $data;

		$this->set_pagination_args( array(
			'total_items' => $total_items,                     // WE have to calculate the total number of items.
			'per_page'    => $per_page,                        // WE have to determine how many items to show on a page.
			'total_pages' => ceil( $total_items / $per_page ), // WE have to calculate the total number of pages.
		) );
	}

	/**
	 * Callback to allow sorting of example data.
	 *
	 * @param string $a First value.
	 * @param string $b Second value.
	 *
	 * @return int
	 */
	protected function usort_reorder( $a, $b ) {
		$a = (array) $a;
		$b = (array) $b;
		// If no sort, default to title.
		$orderby = ! empty( $_REQUEST['orderby'] ) ? wp_unslash( $_REQUEST['orderby'] ) : 'code'; // WPCS: Input var ok.
		// If no order, default to asc.
		$order = ! empty( $_REQUEST['order'] ) ? wp_unslash( $_REQUEST['order'] ) : 'asc'; // WPCS: Input var ok.
		// Determine sort order.
		$result = strnatcmp( $a[ $orderby ], $b[ $orderby ] );

		return ( 'desc' === $order ) ? $result : - $result;
	}
}