<?php

namespace GroundhoggReplacements;

use Groundhogg\Admin\Admin_Page;
use function Groundhogg\get_post_var;
use function Groundhogg\get_request_var;
use function Groundhogg\isset_not_empty;
use \WP_Error;
use function Groundhogg\html;

// Exit if accessed directly
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}


class Replacements_Page extends Admin_Page {
	protected function add_ajax_actions() {
	}

	public function help() {
	}

	public function scripts() {
		wp_enqueue_style('groundhogg-admin');
	}

	protected function add_additional_actions() {
	}

	public function get_slug() {
		return 'gh_replacements';
	}

	public function get_name() {
		return _x( 'Custom Replacements', 'page_title', 'groundhogg' );
	}

	public function get_cap() {
		return 'edit_replacements';
	}

	public function get_item_type() {
		return 'replacement';
	}

	public function get_priority() {
		return 501;
	}

	function get_title() {
		switch ( $this->get_current_action() ) {
			case 'edit':
				return _x( 'Edit Replacement', 'page_title', 'groundhogg' );
				break;
			default:
				return _x( 'Replacements', 'page_title', 'groundhogg' );
				break;
		}
	}

	/**
	 * Process add request
	 * @return bool|WP_Error
	 */
	public function process_add() {
		if ( ! current_user_can( 'add_replacements' ) ) {
			$this->wp_die_no_access();
		}

		if ( ! get_request_var( 'code' ) ) {
			return new \WP_Error( 'no_code', __( 'You must provide a code.', 'groundhogg' ) );
		}

		$code = sanitize_key( get_request_var( 'code' ) );

		if ( ! get_request_var( 'content' ) ) {
			return new \WP_Error( 'no_content', __( 'You must provide content.', 'groundhogg' ) );
		}

		$content     = wp_kses_post( get_post_var( 'content' ) );
		$description = sanitize_textarea_field( get_post_var( 'description' ) );
		$name        = sanitize_textarea_field( get_post_var( 'name' ) );

		$success = Plugin::$instance->custom_replacements->add( $code, $content, $description, $name );

		if ( is_wp_error( $success ) ) {
			return new \WP_Error( $success->get_error_code(), $success->get_error_message() );
		}

		if ( $success ) {
			$this->add_notice( 'created', _x( 'Replacement created.', 'notice', 'groundhogg' ) );
		}

		return true;
	}


	/**
	 * Process edit request.
	 * @return bool|WP_Error
	 */
	public function process_edit() {

		if ( ! current_user_can( 'edit_replacements' ) ) {
			$this->wp_die_no_access();
		}

		if ( isset( $_POST ) ) {

			$code = sanitize_key( get_request_var( 'replacement' ) );

			if ( ! get_request_var( 'content' ) ) {
				return new \WP_Error( 'no_content', __( 'You must provide content.', 'groundhogg-replacements' ) );
			}

			$name        = sanitize_text_field( get_post_var( 'name' ) );
			$content     = wp_kses_post( get_post_var( 'content' ) );
			$description = sanitize_textarea_field( get_post_var( 'description' ) );

			if ( Plugin::$instance->custom_replacements->update( $code, $content, $description, $name ) ) {
				$this->add_notice(
					'created', __( 'Replacement updated', 'groundhogg-replacements' )
				);
			}
		}

		return true;

	}

	/**
	 * Process Delete Request
	 * @return bool
	 */
	public function process_delete() {

		if ( ! current_user_can( 'delete_replacements' ) ) {
			$this->wp_die_no_access();
		}

		foreach ( $this->get_items() as $code ) {
			Plugin::$instance->custom_replacements->delete( $code );
		}

		$this->add_notice(
			'deleted',
			sprintf( _nx( 'Deleted %d replacement', 'Deleted %d replacements', count( $this->get_items() ), 'notice', 'groundhogg-replacements' ),
				count( $this->get_items() )
			)
		);

		return true;
	}

	/**
	 * Display table for shortcode
	 */
	public function table() {
		if ( ! class_exists( 'Replacements_Table' ) ) {
			include dirname( __FILE__ ) . '/replacements-table.php';
		}
		$replacement_table = new Replacements_Table(); ?>
		<div id="col-container" class="wp-clearfix">
			<div id="col-left">
				<div class="col-wrap">
					<div class="form-wrap">
						<h2><?php _e( 'Add New Replacement', 'groundhogg-replacements' ) ?></h2>
						<form id="addreplacement" method="post" action="">
							<input type="hidden" name="action" value="add">
							<?php wp_nonce_field(); ?>
							<div class="form-field">
								<label for="replacement-code-name"><?php _e( 'Name', 'groundhogg-replacements' ) ?></label>
								<input name="name" id="replacement-code-name" type="text" value="" maxlength="100"
								       autocomplete="off" required>
								<p><?php _e( 'What will show in the replacements dropdown.', 'groundhogg-replacements' ) ?></p>
							</div>
							<div class="form-field">
								<label for="replacement-code"><?php _e( 'Replacement Code', 'groundhogg-replacements' ) ?></label>
								<input name="code" id="replacement-code" type="text" value="" maxlength="100"
								       autocomplete="off" required>
								<p><?php _e( 'This is what will be shown in the actual test where you use the code.', 'groundhogg-replacements' ) ?></p>
							</div>
							<div class="form-field">
								<label
									for="replacement-description"><?php _e( 'Description', 'groundhogg-replacements' ) ?></label>
								<?php
								echo html()->textarea( [
									'id'   => 'replacement-description',
									'name' => 'description',
									'rows' => '2'
								] );
								?>
								<p><?php _e( 'This is mostly just for internal use.', 'groundhogg-replacements' ) ?></p>
							</div>
							<div class="form-field">
								<label
									for="replacement-content"><?php _e( 'Content', 'groundhogg-replacements' ) ?></label>
								<p>
								</p>
								<div style="max-width: 700px">
									<?php

									add_action( 'media_buttons', [
										\Groundhogg\Plugin::$instance->replacements,
										'show_replacements_dropdown'
									] );

									wp_editor( '', 'content', [ 'editor_height' => 200, 'editor_width' => 500 ] );
									?>
								</div>

								<p><?php _e( 'Content which will be replaced. Can contain other replacement codes.', 'groundhogg-replacements' ); ?></p>
							</div>
							<?php submit_button( __( 'Add New Replacement', 'groundhogg-replacements' ), 'primary', 'add_replacement' ); ?>
						</form>
					</div>
				</div>
			</div>
			<div id="col-right">
				<div class="col-wrap">
					<form id="posts-filter" method="post">
						<?php $replacement_table->prepare_items(); ?>
						<?php $replacement_table->display(); ?>
					</form>
				</div>
			</div>
		</div>
		<?php
	}

	/**
	 * Loads edit page
	 */
	function edit() {
		if ( ! current_user_can( 'edit_replacements' ) ) {
			$this->wp_die_no_access();
		}
		include dirname( __FILE__ ) . '/edit-replacement.php';
	}

	/**
	 * Process view for the page
	 * @return mixed|void
	 */
	public function view() {
		switch ( $this->get_current_action() ) {
			case 'edit':
				$this->edit();
				break;
			default:
				$this->table();
		}
	}

}