<?php

namespace GroundhoggPro\Steps\Logic;

use Groundhogg\Contact;
use Groundhogg\Step;
use function Groundhogg\add_event_args;
use function Groundhogg\array_any;
use function Groundhogg\array_find;
use function Groundhogg\get_db;
use function Groundhogg\get_event_arg;
use function Groundhogg\html;

class Logic_Loop extends \Groundhogg\Steps\Premium\Logic\Logic_Loop {

	use Trait_Logic_Filters_Basic;

	public function validate_settings( Step $step ) {

		$next = $this->get_setting( 'next' );
		$next = new Step( absint( $next ) );
		$next->merge_changes();

		if ( ! $next->exists() ) {
			$step->add_error( 'loop_broken', 'Please select a step to loop to.' );

			return;
		}

		if ( $next->is_after( $step ) ) {
			$step->add_error( 'loop_broken', 'The target step can\'t come after the loop step.' );

			return;
		}

		if ( ! $next->is_same_branch( $step ) ) {
			$step->add_error( 'loop_broken', 'The target step can\'t be in a different branch than the loop step.' );

			return;
		}

		$limit = $this->get_setting( 'limit' );

		$steps = $step->get_funnel()->get_steps();
		$after = array_find( $steps, function ( Step $other ) use ( $step ) {
			return $other->is_after( $step );
		} );

		$include = $this->get_setting( 'include_filters' );
		$exclude = $this->get_setting( 'exclude_filters' );

		if ( ! $limit && $after && ! $after->is_benchmark() && $after->is_same_branch( $step ) && empty( $include ) && empty( $exclude ) ) {
			$step->add_error( 'cant_continue', 'There are proceeding steps after the loop but they will never run because there is no loop limit or conditions. Add a limit to the number of loops or conditions to solve this.' );

			return;
		}

		if ( ! $this->is_delay_between( $next ) ) {
			$step->add_error( 'cant_continue', 'There are no delay timers or benchmarks within the loop. You might have created an infinite loop by accident.' );
		}
	}

	public function counts_as_delay( Step $step ) {
		return $step->is_timer() || $step->is_benchmark();
	}

	public function is_delay_between( Step $from ) {

		$to    = $this->get_current_step();
		$steps = $to->get_funnel()->get_steps();

		// inclusive
		if ( $this->counts_as_delay( $from ) ) {
			return true;
		}

		return array_any( $steps, function ( Step $other ) use ( $from, $to ) {
			return $other->is_after( $from ) && $other->is_before( $to ) && $this->counts_as_delay( $other );
		} );
	}

	protected function settings_should_ignore_morph() {
		return false;
	}

	public function sortable_item( $step ) {
		parent::sortable_item( $step ); // TODO: Change the autogenerated stub
	}

	/**
	 * @param Step $step
	 *
	 * @return void
	 */
	public function settings( $step ) {

		echo html()->e( 'p', [], __( 'If the contact matches the conditions (leave conditions empty to loop all contacts)...' ) );

		echo html()->e( 'div', [ 'class' => 'include-search-filters ignore-morph' ], [ html()->e( 'div', [ 'id' => $this->setting_id_prefix( 'include_filters' ) ] ) ] );
		echo html()->e( 'div', [ 'class' => 'exclude-search-filters ignore-morph' ], [ html()->e( 'div', [ 'id' => $this->setting_id_prefix( 'exclude_filters' ) ] ) ] );

		$preceding = array_filter( $step->get_funnel()->get_steps(), function ( $preceding ) use ( $step ) {
			return $preceding->is_same_branch( $step ) && $preceding->is_before( $step ) && $preceding->is_action() && $this->is_delay_between( $preceding );
		} );

		if ( empty( $preceding ) ) {
			// can only loop to previous steps if there is a delay in between

			echo html()->e( 'p', [], __( 'You can only loop to actions that are in the <b>same branch</b> as the <span class="gh-text purple bold">loop logic</span>. There must also be a delay (either a <span class="gh-text green bold">timer delay</span> or a <span class="gh-text orange bold">benchmark</span>) between the <span class="gh-text purple bold">loop logic</span> and the <span class="gh-text green bold">action</span> you are looping to prevent accidental infinite looping.' ) );
			echo html()->e( 'p', [], __( 'Add additional preceding <span class="gh-text green bold">actions</span> and a delay to the branch to select an action to loop to.' ) );

			return;
		}

		echo html()->e( 'p', [], __( 'Then loop back to...', 'groundhogg-pro' ) );

		$options = [];

		foreach ( $preceding as $_s ) {
			$options[ $_s->get_id() ] = sprintf( '%d. %s', $_s->get_order(), sanitize_text_field( $_s->get_title() ) );
		}

		$limit = absint( $this->get_setting( 'limit' ) );

		if ( $limit < 1 ) {
			$limit = '';
		}

		echo html()->e( 'div', [
			'class' => 'display-flex gap-5 align-center'
		], [

			html()->select2( [
				'name'        => $this->setting_name_prefix( 'next' ),
				'options'     => $options,
				'selected'    => $this->get_setting( 'next' ),
				'option_none' => false
			] ),

			html()->input( [
				'type'        => 'number',
				'class'       => 'number',
				'min'         => 1,
				'name'        => $this->setting_name_prefix( 'limit' ),
				'value'       => $limit,
				'placeholder' => 'Unlimited'
			] ),

			html()->e( 'span', [], 'time(s).' )

		] );

		?><p></p><?php
	}

	public function get_settings_schema() {
		return [
			'include_filters' => [
				'default'  => [],
				'initial'  => [],
				'sanitize' => [ $this, 'sanitize_filters' ],
			],
			'exclude_filters' => [
				'default'  => [],
				'initial'  => [],
				'sanitize' => [ $this, 'sanitize_filters' ],
			],
			'next'            => [
				'sanitize' => function ( $value ) {
					$step = new Step( absint( $value ) );
					if ( ! $step->exists() || ! $this->is_delay_between( $step ) ) {
						return 0;
					}

					return $step->ID;
				},
				'import'   => 'absint',
				'default'  => 0,
			],
			'limit'           => [
				'sanitize' => function ( $value ) {
					$int = absint( $value );
					if ( $int > 1 ) {
						return $int;
					}

					return '';
				},
				'default'  => 0
			]
		];
	}

	/**
	 * update the next if defined
	 *
	 * @param $step Step
	 *
	 * @return void
	 */
	public function post_import( $step ) {

		// This will be the ID of the imported step
		$old_next = absint( $step->get_meta( 'next' ) );

		if ( ! $old_next ) {
			return;
		}

		// the latest one will always be the most recently imported
		$meta = get_db( 'stepmeta' )->query( [
			'meta_key'   => 'imported_step_id',
			'meta_value' => $old_next,
			'limit'      => 1,
			'orderby'    => 'step_id',
			'order'      => 'desc'
		] );

		$step_id = $meta[0]->step_id;

		$step->update_meta( 'next', $step_id );

	}

	public function generate_step_title( $step ) {

		$next = new Step( $this->get_setting( 'next' ) );

		if ( ! $next->exists() ) {
			return 'Loop';
		}

		$limit = $this->get_setting( 'limit' );

		$title = sprintf( 'Loop to <b>%s</b>', $next->get_title() );

		if ( $limit ) {

			switch ( absint( $limit ) ) {
				case 1:
					$title .= ' once';
					break;
				case 2:
					$title .= ' twice';
					break;
				default:
					$title .= " $limit times";
					break;
			}
		}

		return $title;
	}

	public function get_logic_action( Contact $contact ) {

		if ( ! $this->matches_filters( $contact ) ) {
			return false;
		}

		$current    = $this->get_current_step();
		$times_args = $current->ID . '-times';

		// check to make sure we are not passed the limit
		$limit = absint( $this->get_setting( 'limit' ) );
		$times = absint( get_event_arg( $times_args, 0 ) );

		// if we have exhausted the limit of the loop
		if ( $limit >= 1 && $times >= $limit ) {
			return false;
		}

		$_next = absint( $this->get_setting( 'next' ) );

		if ( ! $_next ) {
			return false;
		}

		$_next = new Step( $_next );

		if ( ! $_next->exists() ) {
			return false;
		}

		if ( $_next->is_same_branch( $this->get_current_step() ) && $_next->is_before( $this->get_current_step() ) ) {

			add_event_args( [
				$times_args => $times + 1
			] );

			return $_next;
		}

		return false;
	}
}
