<?php

namespace GroundhoggPro\Steps\Actions;

use Groundhogg\Admin\Funnels\Simulator;
use Groundhogg\Contact;
use Groundhogg\Event;
use Groundhogg\Step;
use function Groundhogg\array_bold;
use function Groundhogg\bold_it;
use function Groundhogg\html;
use function Groundhogg\orList;

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Apply Owner
 *
 * Apply an owner through the funnel builder depending on the the funnel
 *
 * @since       File available since Release 0.9
 * @subpackage  Elements/Actions
 * @author      Adrian Tobey <info@groundhogg.io>
 * @copyright   Copyright (c) 2018, Groundhogg Inc.
 * @license     https://opensource.org/licenses/GPL-3.0 GNU Public License v3
 * @package     Elements
 */
class Apply_Owner extends \Groundhogg\Steps\Premium\Actions\Apply_Owner {

	/**
	 * @param $step Step
	 */
	public function settings( $step ) {

		echo html()->e( 'p', [], __( 'Selecting one or more users to assign as the contact owner. Selecting multiple owners will initiate a <i>round robin</i>.', 'groundhogg-pro' ) );

		echo html()->dropdown( [
			'id' => $this->setting_id_prefix( 'owners' )
		] )

		?><p></p><?php
	}

	public function generate_step_title( $step ) {

		$owner_ids = wp_parse_id_list( $this->get_setting( 'owner_id' ) );
		$owners    = array_map( 'get_userdata', $owner_ids );

		return sprintf( 'Assign to %s', orList( array_bold( wp_list_pluck( $owners, 'display_name' ) ) ) );
	}

	public function get_settings_schema() {
		return [
			'owner_id' => [
				'sanitize' => function ( $value ) {

					if ( ! is_array( $value ) ) {
						return [];
					}

					return array_filter( wp_parse_id_list( $value ), function ( $id ) {
						return user_can( $id, 'view_contacts' );
					} );
				},
				'default'  => [],
				'initial'  => []
			],
			'index'    => [
				'default' => 0,
				'initial' => 0,
                'sanitize' => 'absint'
			]
		];
	}

	/**
	 * Process the apply owner step...
	 *
	 * @param $contact Contact
	 * @param $event   Event
	 *
	 * @return bool
	 */
	public function run( $contact, $event ) {

		$owners = $this->get_setting( 'owner_id' );
		$owners = is_array( $owners ) ? wp_parse_id_list( $owners ) : [ absint( $owners ) ];

        // no owners
        if ( empty( $owners ) ) {
	        Simulator::log( "❌ No users were defined."  );
            return false;
        }

		if ( count( $owners ) === 1 ) { // only one owner selected

            $owner_id = $owners[0];

		} else { // round-robin

			// can't use save_setting/update_meta because of publish changes feature ugh...
			$step     = $this->get_current_step();
			$i        = absint( $step->get_meta( 'index' ) );
			$owner_id = $owners[ $i ];

			$i ++;

			if ( $i >= count( $owners ) ) {
				$i = 0;
			}

			// don't let changes scare us away
			$step->update_meta_bypass_changes( 'index', $i );
		}

		$changed = $contact->change_owner( $owner_id );

        if ( ! $changed ){

	        Simulator::log( "❌ Owner was not changed."  );

            return false;
        }

		Simulator::log( sprintf( "🧑🏻‍💼 Set owner to %s", bold_it( $contact->get_ownerdata()->display_name ) ) );

		return true;
	}
}
