<?php

namespace GroundhoggPro;

use Groundhogg\Admin\Funnels\Simulator;
use Groundhogg\Contact;
use Groundhogg\Step;
use function Groundhogg\add_event_args;
use function Groundhogg\get_event_arg;

class Handle_Timers {

	public function __construct() {
		add_filter( 'groundhogg/step/enqueue', [ $this, 'filter_timers' ], 100, 2 );

		// loops and skips...
		add_filter( 'groundhogg/step/next_action', [ $this, 'filter_loop_next' ], 101, 3 );
		add_filter( 'groundhogg/step/next_action', [ $this, 'filter_skip_next' ], 101, 3 );
	}

	/**
	 * Handle the looping functionality
	 *
	 * @param         $next
	 * @param Step    $step
	 * @param Contact $contact
	 *
	 * @return Step
	 */
	function filter_loop_next( $next, Step $step, Contact $contact ) {

		if ( ! $step->type_is( 'loop' ) ) {
			return $next;
		}

		Simulator::log( '◆ Evaluating loop...' );

		$times_args = $step->ID . '-times';

		// check to make sure we are not passed the limit
		$limit = absint( $step->get_meta( 'limit' ) );
		$times = absint( get_event_arg( $times_args, 0 ) );

		// if we have exhausted the limit of the loop
		if ( $limit >= 1 && $times >= $limit ) {
			return $next;
		}

		$_next = absint( $step->get_meta( 'next' ) );

		if ( ! $_next ) {
			Simulator::log( '↳ No target step, passing through...' );

			return $next;
		}

		add_event_args( [
			$times_args => $times + 1,
		] );

		$_next = new Step( $_next );

		if ( $_next->exists() ) {
			Simulator::log( '↳ Looping to target...' );

			return $_next;
		}

		Simulator::log( '↳ No target step, passing through...' );

		return $next;
	}

	/**
	 * Handle the looping functionality
	 *
	 * @param         $next
	 * @param Step    $step
	 * @param Contact $contact
	 *
	 * @return Step
	 */
	function filter_skip_next( $next, Step $step, Contact $contact ) {

		if ( ! $step->type_is( 'skip' ) ) {
			return $next;
		}

		Simulator::log( '◆ Evaluating skip...' );

		$_next = absint( $step->get_meta( 'next' ) );

		if ( ! $_next ) {
			Simulator::log( '↳ No target step, passing through...' );

			return $next;
		}

		$_next = new Step( $_next );

		if ( $_next->exists() ) {
			Simulator::log( '↳ Skipping to target...' );

			return $_next;
		}

		Simulator::log( '↳ No target step, passing through...' );

		return $next;
	}

	/**
	 * Filter the step directly because of timer logic
	 *
	 * @param Step    $step can't be strongly typed!!!!
	 * @param Contact $contact
	 *
	 * @return false|Step
	 */
	public function filter_timers( $step, Contact $contact ) {

		if ( ! is_a( $step, Step::class ) ) {
			return $step;
		}

		// not a timer with logic
		if ( ! in_array( $step->get_type(), [
			'date_timer',
			'advanced_timer',
			'field_timer',
		] ) ) {
			return $step;
		}

		Simulator::log( '◆ Evaluating timer logic...' );

		// just in case...
		$step->enqueued_contact = $contact;

		// just make sure...
		if ( $step->get_run_time() > time() ) {
			Simulator::log( '↳ Date in the future...' );

			return $step;
		}

		Simulator::log( '↳ Date is in the past...' );

		// If the timer was a date in the past, the execution time will be modified to a minute less than the current system time.
		// This will mean the scheduled time will be less than the enqueued time still and all is well.
		$date_passed = $step->get_meta( 'date_passed' ) ?: 'passthru';

		switch ( $date_passed ) {

			// Just queue up the next step instead
			default:
			case 'passthru':

				$next_step = $step->get_next_action( $contact );

				if ( $next_step instanceof Step ) {
					Simulator::log( '↳ Passing through to the next step...' );

					return $next_step;
				}

				return false;

			// Skip to another step.
			case 'skip_to':

				$skip_to_step_id = absint( $step->get_meta( 'skip_to_step' ) );

				$step = new Step( $skip_to_step_id );

				if ( $step->exists() ) {
					Simulator::log( '↳ Skipping to target step...' );

					return $step;
				}

				return false;


			// Stop the funnel
			case 'stop':
				Simulator::log( '■ Stopping the flow...' );

				// do nothing...
				return false;

		}
	}
}
