<?php

namespace GroundhoggGForms\Steps;

use Groundhogg\Contact;
use Groundhogg\Steps\Benchmarks\Form_Integration;
use function Groundhogg\after_form_submit_handler;
use function Groundhogg\generate_contact_with_map;
use function Groundhogg\get_array_var;
use function Groundhogg\isset_not_empty;
use function Groundhogg\words_to_key;

class GForm extends Form_Integration {

	public function get_name() {
		return __( 'Gravity Forms', 'groundhogg-gravity' );
	}

	public function get_type() {
		return 'gravity_form';
	}

	public function get_description() {
		return __( 'Integrate Groundhogg with Gravity Forms.', 'groundhogg-gravity' );
	}

	public function get_icon() {
		return GROUNDHOGG_GFORMS_ASSETS_URL . 'images/gravity-forms.svg';
	}

	protected function get_complete_hooks() {
		return [
			'gform_after_submission' => 2
		];

	}

	/**
	 * @param $entry
	 * @param $form
	 */
	public function setup( $entry, $form ) {
		$this->add_data( 'form_id', absint( $form['id'] ) );
		$this->add_data( 'posted_data', $this->normalize_posted_data( $entry, $form ) );
		$this->add_data( 'posted_data_old_format', $entry );
	}

	/**
	 * Normalize the data to words_to_key( label ) instead of the field id e.g. 1.1
	 *
	 * @param $fields array
	 *
	 * @return array
	 */
	private function normalize_posted_data( $entry, $form ) {

		$normalized = [];

		$normalized_id = function ( $field, $input_id = false ) {
			$label = \GFCommon::get_label( $field, $input_id );
			$id    = $input_id ?: (string) $field["id"];

			return words_to_key( $label . '_' . str_replace( '.', '_', $id ) );
		};

		$add_to_normalized = function ( $field, $value, $input_id = false ) use ( $normalized_id, &$normalized ) {
			$normalized[ $normalized_id( $field, $input_id ) ] = $value;
		};

		// Something went wrong here
		if ( ! is_array( $form['fields'] ) ) {
			return [];
		}

		foreach ( $form["fields"] as $field ) {

			switch ( $field['type'] ) {
				case 'checkbox':

					$checked = [];

					foreach ( $field["inputs"] as $input ) {
						$value = get_array_var( $entry, $input['id'] );
						if ( $value ) {
							$checked[] = $value;
						}
					}

					$add_to_normalized( $field, $checked );
					break;
				case 'list':
					$value = maybe_unserialize( get_array_var( $entry, $field['id'] ) );
					$add_to_normalized( $field, $value );

					break;
				case 'fileupload':

					$value      = get_array_var( $entry, $field['id'] );
					$maybe_json = json_decode( $value, true );

					// was json encoded array
					if ( $maybe_json ) {
						$value = $maybe_json;
					}

					$add_to_normalized( $field, $value );

					break;
				case 'name':
				case 'address':
					foreach ( $field["inputs"] as $input ) {
						$value = get_array_var( $entry, $input['id'] );
						$add_to_normalized( $field, $value, $input['id'] );
					}
					break;
				default:

					// ignore display only
					if ( rgar( $field, 'displayOnly' ) ) {
						break;
					}

					// check to see if value exists
					$value = get_array_var( $entry, $field['id'] );

					// value is empty and field has inputs, check those
					if ( ! $value && isset_not_empty( $field, 'inputs' ) ) {
						foreach ( $field["inputs"] as $input ) {
							$value = get_array_var( $entry, $input['id'] );
							$add_to_normalized( $field, $value, $input['id'] );
						}
						break;
					}

					// if here, value exists
					$add_to_normalized( $field, $value );

					break;
			}
		}

		return $normalized;
	}

	/**
	 * Generate a contact from the map.
	 *
	 * If the setting to use the new format is set, use the nw format. Otherwise
	 * keep using the old format.
	 *
	 * @return false|Contact
	 */
	public function get_the_contact() {

		// SKIP if not the right form.
		if ( ! $this->can_complete_step() ) {
			return false;
		}

		if ( $this->get_setting( 'use_new_format' ) ) {
			$posted_data = $this->get_data( 'posted_data' );
		} else {
			$posted_data = $this->get_data( 'posted_data_old_format' );
		}

		$field_map = $this->get_setting( 'field_map' );

		$contact = generate_contact_with_map( $posted_data, $field_map );

		if ( ! $contact || is_wp_error( $contact ) ) {
			return false;
		}

		after_form_submit_handler( $contact );

		return $contact;
	}

	/**
	 * Get forms for the select 2 picker
	 *
	 * @return array
	 */
	protected function get_forms_for_select_2() {
		$forms = \GFFormsModel::get_forms( true );

		$data = [];

		if ( $forms ) {
			foreach ( $forms as $form ) {
				$data[ $form->id ] = $form->title;
			}
		}

		return $data;
	}

	/**
	 * @param $form_id
	 *
	 * @return array
	 */
	protected function get_form_fields( $form_id ) {
		$form            = \RGFormsModel::get_form_meta( $form_id );
		$mappable_fields = [];

		// Something went wrong here
		if ( ! is_array( $form['fields'] ) ) {
			return [];
		}

		$add_mappable_field = function ( $id, $label ) use ( &$mappable_fields ) {
			$mappable_fields[] = [ $id, $label ];
		};

		foreach ( $form["fields"] as $field ) {

			switch ( $field['type'] ) {

				case 'name':
				case 'address':
					foreach ( $field["inputs"] as $input ) {
						$add_mappable_field( $input['id'], \GFCommon::get_label( $field, $input["id"] ) );
					}
					break;

				case 'checkbox':
				case 'list':
				case 'fileupload':
				case 'time':
				case 'date':
				case 'email':
					$add_mappable_field( $field['id'], \GFCommon::get_label( $field ) );
					break;
				default:

					// ignore display only
					if ( rgar( $field, 'displayOnly' ) ) {
						break;
					}

					if ( isset_not_empty( $field, 'inputs' ) ){
						foreach ( $field["inputs"] as $input ) {
							$add_mappable_field( $input['id'], \GFCommon::get_label( $field, $input["id"] ) );
						}
						break;
					}

					$add_mappable_field( $field['id'], \GFCommon::get_label( $field ) );

					break;
			}
		}

		return $mappable_fields;
	}

	/**
	 * @param int|string   $key
	 * @param array|string $field
	 *
	 * @return array
	 */
	protected function normalize_field( $key, $field ) {
		return [
			'id'    => words_to_key( $field[1] . '_' . str_replace( '.', '_', $field[0] ) ),
			'label' => $field[1]
		];
	}

	/**
	 * Once saved use the new format.
	 *
	 * If not saved do not use new format.
	 *
	 * @param \Groundhogg\Step $step
	 */
	public function save( $step ) {
		$this->save_setting( 'use_new_format', true );
		parent::save( $step );
	}
}
