<?php

namespace GroundhoggContentRestriction;

use Groundhogg\Contact_Query;
use function Groundhogg\get_contactdata;
use function Groundhogg\get_request_var;

class Restriction {

	public function __construct() {
		add_action( 'template_redirect', [ $this, 'view_protect' ] );
		add_action( 'pre_get_posts', [ $this, 'query_protect' ], 9999 );
	}

	/**
	 * @return array
	 */
	protected function get_restricted_content() {
		return array_values( get_option( 'gh_restricted_posts', [] ) );
	}

	public function query_protect( $query ) {

		if ( ! isset( $query ) ) {
			return;
		}

		if ( $query->is_singular || ( function_exists( 'is_bbpress' ) && is_bbpress() ) || is_admin() ) {
			return;
		}

		if ( current_user_can( 'edit_posts' ) ) {
			return;
		}

		// hide downloads from all queries except singular pages, which will 404 without the conditional
		// is_singular('download') doesn't work inside pre_get_posts
		if ( ! $query->is_singular ) {
			$excluded_ids = isset( $query->query_vars['post__not_in'] ) ? $query->query_vars['post__not_in'] : [];

			$ids = $this->get_restricted_content();

			foreach ( $ids as $id ) {
				if ( $this->is_post_restricted( $id ) ) {
					$excluded_ids[] = $id;
				}
			}

			// make sure we're merging with existing post__not_in so we do not override it
			$query->set( 'post__not_in', $excluded_ids );
		}
	}

	/**
	 * @param $post_id
	 *
	 * @return true|false true if the content is restricted, false other wise.
	 */
	public function is_post_restricted( $post_id ) {
		$is_restricted = get_post_meta( $post_id, 'groundhogg_content_restriction_enable', true );

		if ( ! $is_restricted ) {
			return false;
		}

		$login_in_required = get_post_meta( $post_id, 'groundhogg_content_restriction_require_login', true );

		if ( $login_in_required && ! is_user_logged_in() ) {
			return true;
		}

		$contact = get_contactdata();

		if ( ! $contact ) {
			return true;
		}

		$filters = get_post_restriction_filters( $post_id );

		// No filters were defined, so we are going to ignore...
		if ( empty( $filters ) ) {
			return false;
		}

		// the Groundhogg query assumes sales manager role and will add the owner id clause when it is not needed.
		// Remove it!
		add_action( 'gh_parse_contact_query', function ( $query ) {
			unset( $query->query_vars['owner'] );
		} );

		$query = new Contact_Query();

		$count = $query->count( [
			'filters' => $filters,
			'include' => [ $contact->get_id() ]
		] );

		// If the count is equal to 0 the post is restricted
		return $count === 0;
	}

	/**
	 * Redirect to the url provided.
	 *
	 * @param $post_id int
	 */
	public function content_restriction_redirect( $post_id ) {
		$url = get_post_meta( $post_id, 'groundhogg_content_restriction_redirect', true );

		if ( ! $url ) {
			$url = get_option( 'gh_content_restriction_default_redirect', home_url() );
		}

		wp_redirect( $url ?: home_url() );
		die();
	}

	/**
	 * If the content is restricted, redirect!
	 */
	public function view_protect() {

		$ids = $this->get_restricted_content();

		if ( ! is_singular() || ! get_the_ID() || current_user_can( 'edit_posts' ) ) {
			return;
		}

		$post_id = get_the_ID();

		if ( in_array( $post_id, $ids ) && $this->is_post_restricted( $post_id ) ) {

			$this->content_restriction_redirect( $post_id );
		}
	}

}