<?php

namespace GroundhoggContentRestriction\Admin;

use GroundhoggContentRestriction\Api\Content_Restriction_Api;
use function Groundhogg\get_request_var;
use function Groundhogg\html;
use function GroundhoggContentRestriction\get_post_restriction_filters;

class Metabox {

	public function __construct() {
		add_action( 'add_meta_boxes', [ $this, 'register' ] );
		add_action( 'save_post', [ $this, 'save' ] );
	}

	public function register() {

		if ( ! current_user_can( 'view_contacts' ) ) {
			return;
		}

		add_meta_box(
			'groundhogg_content_restriction',
			__( 'Content Restriction', 'groundhogg-content-restriction' ),
			[ $this, 'render' ],
			null,
			'normal',
			'default'
		);
	}

	public function render( $post ) {

		$post_id = $post->ID;

		wp_enqueue_script( 'groundhogg-content-restriction-admin' );
		wp_localize_script( 'groundhogg-content-restriction-admin', 'GroundhoggContentRestriction', [
			'post'       => $post,
			'rest_route' => rest_url( Content_Restriction_Api::NAME_SPACE . '/content-restriction' ),
			'filters'    => get_post_restriction_filters( $post_id ),
			'_old'       => [
				'tags'         => wp_parse_id_list( get_post_meta( $post_id, 'groundhogg_content_restriction_tags', true ) ),
				'relationship' => get_post_meta( $post_id, 'groundhogg_content_restriction_tags_relationship', true )
			]
		] );

		wp_nonce_field( 'groundhogg_content_restriction_save', 'groundhogg_content_restriction' );

		echo html()->wrap( html()->wrap( __( 'Enable Content Restriction', 'groundhogg-content-restriction' ), 'b' ), 'p' );

		echo html()->checkbox( [
			'label'   => __( 'Enable', 'groundhogg-content-restriction' ),
			'name'    => 'groundhogg_content_restriction_enable',
			'id'      => 'groundhogg_content_restriction_enable',
			'class'   => '',
			'value'   => '1',
			'checked' => get_post_meta( $post_id, 'groundhogg_content_restriction_enable', true ),
			'title'   => __( 'Enable content restriction', 'groundhogg-content-restriction' ),
		] );

		echo html()->description( __( 'Whether to restrict this content.', 'groundhogg-content-restriction' ) );

		echo html()->wrap( html()->wrap( __( 'Require Login', 'groundhogg-content-restriction' ), 'b' ), 'p' );

		echo html()->checkbox( [
			'label'   => __( 'Enable', 'groundhogg-content-restriction' ),
			'name'    => 'groundhogg_content_restriction_require_login',
			'id'      => 'groundhogg_content_restriction_require_login',
			'class'   => '',
			'value'   => '1',
			'checked' => get_post_meta( $post_id, 'groundhogg_content_restriction_require_login', true ),
			'title'   => __( 'Enable require login', 'groundhogg-content-restriction' ),
		] );

		echo html()->description( __( 'Does the current contact need to be logged in?', 'groundhogg-content-restriction' ) );

		echo html()->wrap( html()->wrap( __( 'Filtering Rules', 'groundhogg-content-restriction' ), 'b' ), 'p' );

		?>
		<div id="gh-content-restriction"></div>
		<div id="gh-content-restriction-total-preview"></div>
		<?php

		echo html()->wrap( html()->wrap( __( 'Redirect', 'groundhogg-content-restriction' ), 'b' ), 'p' );

		echo html()->link_picker( [
			'name'  => 'groundhogg_content_restriction_redirect',
			'id'    => 'groundhogg_content_restriction_redirect',
			'style' => [ 'max-width' => '100%' ],
			'value' => get_post_meta( $post_id, 'groundhogg_content_restriction_redirect', true )
		] );

		echo html()->description( __( 'Where to send traffic if they do not meet the requirements.', 'groundhogg-content-restriction' ) );

	}

	/**
	 * Save the settings.
	 * The actual filters are now saved via the REST API and not via $_POST.
	 *
	 * @param $post_id
	 */
	public function save( $post_id ) {

		if ( ! current_user_can( 'view_contacts' ) ) {
			return;
		}

		if ( ! wp_verify_nonce( get_request_var( 'groundhogg_content_restriction' ), 'groundhogg_content_restriction_save' ) ) {
			return;
		}

		$enabled   = boolval( get_request_var( 'groundhogg_content_restriction_enable', false ) );
		$logged_in = boolval( get_request_var( 'groundhogg_content_restriction_require_login', false ) );

		update_post_meta( $post_id, 'groundhogg_content_restriction_enable', $enabled );
		update_post_meta( $post_id, 'groundhogg_content_restriction_require_login', $logged_in );
		update_post_meta( $post_id, 'groundhogg_content_restriction_redirect', sanitize_text_field( get_request_var( 'groundhogg_content_restriction_redirect', site_url() ) ) );

		$restricted_posts = get_option( 'gh_restricted_posts', [] );

		if ( $enabled && ! in_array( $post_id, $restricted_posts ) ) {
			$restricted_posts[ $post_id ] = $post_id;
		} else if ( ! $enabled ) {
			unset( $restricted_posts[ $post_id ] );
		}

		update_option( 'gh_restricted_posts', $restricted_posts );
	}

}