<?php

namespace GroundhoggLogic;

use Groundhogg\Admin\Funnels\Simulator;
use Groundhogg\Contact;
use Groundhogg\Contact_Query;
use Groundhogg\Queue\Event_Queue;
use Groundhogg\Step;
use Groundhogg\Steps\Funnel_Step;
use function Groundhogg\dashicon_e;
use function Groundhogg\enqueue_filter_assets;
use function Groundhogg\get_array_var;
use function Groundhogg\get_db;
use function Groundhogg\get_url_var;
use function Groundhogg\isset_not_empty;

class Logic {

	public function __construct() {

		add_action( 'admin_enqueue_scripts', [ $this, 'scripts' ] );
		add_action( 'groundhogg/steps/sortable/labels', [ $this, 'maybe_show_logic_helper' ], 99, 1 );
//		add_action( 'groundhogg/steps/sortable/inside', [ $this, 'maybe_show_action_logic_above' ], 99, 1 );
		add_filter( 'groundhogg/steps/sortable/classes', [ $this, 'maybe_add_has_logic_class' ], 99, 3 );

		add_action( 'groundhogg/steps/settings/after', [ $this, 'wrap_render' ], 99, 1 );

//        add_action( 'groundhogg/steps/sortable/inside', [ $this ] );

		// Handlers
//		add_filter( 'groundhogg/steps/enqueue', [ $this, 'handle_step_enqueue' ], 99, 3 );
		add_filter( 'groundhogg/step/enqueue', [ $this, 'filter_step' ], 99, 2 );

		add_action( 'groundhogg/step/post_setup', [ $this, 'maybe_upgrade_logic_conditions' ] );

		add_action( 'groundhogg/steps/post_import', [ $this, 'logic_imported' ] );

//        add_filter( 'groundhogg/step/next_action', [ $this, 'handle_next_action' ] );
	}

	public function maybe_add_has_logic_class( $classes, $step, $type ) {

		$logic_enabled = boolval( $step->get_meta( '_conditional_logic_enabled' ) );

		// Not enabled? Return out;
		if ( ! $logic_enabled || ! $step->is_action() ) {
			return $classes;
		}

		$classes[] = 'has-conditional-logic';

		return $classes;
	}

	/**
	 * Maybe fix logic skip_to setting
	 *
	 * @param $step Step
	 *
	 * @return void
	 */
	public function logic_imported( $step ) {

		// benchmarks dont have skip_to
		if ( $step->is_benchmark() ) {
			return;
		}

		$logic = (array) $step->get_meta( '_conditional_logic' );

		if ( ! $logic ) {
			return;
		}

		$skip_to = absint( get_array_var( $logic, 'skip_to' ) );

		if ( ! $skip_to ) {
			return;
		}

		// the latest one will always be the most recently imported
		$meta = get_db( 'stepmeta' )->query( [
			'meta_key'   => 'imported_step_id',
			'meta_value' => $skip_to,
			'limit'      => 1,
			'orderby'    => 'step_id',
			'order'      => 'desc'
		] );

		$step_id = $meta[0]->step_id;

		$logic['skip_to'] = $step_id;

		$step->update_meta( '_conditional_logic', $logic );
	}

	/**
	 * Maybe upgrade logic to new format
	 *
	 * @param $step Step
	 */
	public function maybe_upgrade_logic_conditions( $step ) {

		$logic = $step->get_meta( '_conditional_logic' );

		// If empty, or filters are configured
		if ( empty( $logic )
		     || ! isset_not_empty( $logic, 'relationship' )
		     || ! isset_not_empty( $logic, 'conditions' ) ) {
			return;
		}

		$filters         = [];
		$exclude_filters = [];

		$relationship = strtolower( get_array_var( $logic, 'relationship', 'or' ) );
		$conditions   = get_array_var( $logic, 'conditions', [] );

		$_filters = [];

		$compare_map = [
			'='         => 'equals',
			'!='        => 'not_equals',
			'>'         => 'greater_than',
			'less_than' => 'less_than',
			'in'        => 'contains',
			'!in'       => 'not_contains',
			'_'         => 'empty',
			'!_'        => 'not_empty',
		];

		foreach ( $conditions as $condition ) {
			$type   = get_array_var( $condition, 'type' );
			$config = get_array_var( $condition, 'config' );

			$filter = [];

			switch ( $type ) {
				case 'tags':

					$filter = [
						'type'     => 'tags',
						'compare'  => get_array_var( $config, 'when' ) === 'none' ? 'excludes' : 'includes',
						'compare2' => get_array_var( $config, 'when' ) === 'any' ? 'any' : 'all',
						'tags'     => wp_parse_id_list( get_array_var( $config, 'tags' ) )
					];

					break;

				case 'meta':

					$filter = [
						'type'    => 'meta',
						'meta'    => sanitize_key( get_array_var( $config, 'key' ) ),
						'compare' => strtolower( $compare_map[ get_array_var( $config, 'compare' ) ] ),
						'value'   => get_array_var( $config, 'value' ),
					];

					break;

				case 'contact':

					$key     = sanitize_key( get_array_var( $config, 'key' ) );
					$compare = get_array_var( $config, 'compare' );
					$value   = get_array_var( $config, 'value' );

					switch ( $key ) {
						case 'first_name':
						case 'last_name':
						case 'email':
							$filter = [
								'type'    => $key,
								'compare' => $compare_map[ $compare ],
								'value'   => $value
							];
							break;
						case 'owner_id':
							$filter = [
								'type'    => 'owner',
								'compare' => $compare == '!=' ? 'not_in' : 'in',
								'value'   => [ $value ]
							];
							break;
						case 'optin_status':
							$filter = [
								'type'    => 'optin_status',
								'compare' => $compare == '!=' ? 'not_in' : 'in',
								'value'   => [ $value ]
							];
							break;
						case 'date_created':
							$filter = [
								'type'                                => 'date_created',
								'date_range'                          => $compare === '>' ? 'after' : 'before',
								$compare === '>' ? 'after' : 'before' => $value
							];
							break;
					}
			}

			$_filters[] = $filter;
		}

		switch ( $relationship ) {
			default:
			case 'or':
				$filters = array_map( function ( $filter ) {
					return [ $filter ];
				}, $_filters );
				break;
			case 'and':
				$filters = [ $_filters ];
				// All conditions must pass
				break;
			case 'none':
				$exclude_filters = [ $_filters ];
				break;

		}

		$skip_to = absint( get_array_var( $logic, 'skip_to' ) );
		$result  = get_array_var( $logic, 'result' );

		$step->update_meta( '_conditional_logic', [
			'filters'         => $filters,
			'exclude_filters' => $exclude_filters,
			'result'          => $result,
			'skip_to'         => $skip_to
		] );
	}


	/**
	 * @param $step Step
	 *
	 * @return void
	 */
	public function maybe_show_action_logic_above( $step ) {

		// Check if conditional logic is enabled
		$logic_enabled = boolval( $step->get_meta( '_conditional_logic_enabled' ) );

		// Not enabled? Return out;
		if ( ! $logic_enabled || ! $step->is_action() ) {
			return;
		}

		$logic  = $step->get_meta( '_conditional_logic' );
		$result = get_array_var( $logic, 'result' );

		?>
        <div class="logic-above">
            <div class="if-condition-wrap">
                <span class="if-condition"><span class="if-label">If...</span></span>
                <div class="logic-display" data-id="<?php echo $step->ID; ?>"></div>
            </div>
			<?php

			// can't skip if it's the last step.
			if ( $step->is_last() ) {
				$result = 'stop';
			}

			switch ( $result ) {
				case 'stop':
					?><span class="stop-funnel">
					<?php dashicon_e( 'no-alt' ); ?>
                    <span class="gh-tooltip top">
                        <?php _e( 'Stop the funnel' ) ?>
                    </span>
                    </span><?php
					break;
				default:
				case 'skip':
					?>
                    <div class="skip-next"></div><?php
					break;
				case 'skip_to':
					$skip_to = absint( get_array_var( $logic, 'skip_to' ) );

					?>
                <div class="skip-to" data-id="<?php echo $skip_to ?>"></div><?php
					break;
			}

			?>
        </div>
		<?php
	}

	/**
	 * @param $step Step
	 *
	 * @return void
	 */
	public function maybe_show_logic_helper( $step ) {
		// Check if conditional logic is enabled
		$logic_enabled = boolval( $step->get_meta( '_conditional_logic_enabled' ) );

		// Not enabled? Return out;
		if ( ! $logic_enabled || $step->is_logic() ) {
			return;
		}

		?>
        <span class="logic-tag">
            <div class="logic-display" data-id="<?php echo $step->ID; ?>"></div>
            <?php dashicon_e( 'filter' ); ?>
            </span>
		<?php

	}

	public function scripts() {
		if ( get_url_var( 'page' ) !== 'gh_funnels' || get_url_var( 'action' ) !== 'edit' ) {
			return;
		}

		enqueue_filter_assets();

		wp_enqueue_style( 'groundhogg-logic-admin' );
		wp_enqueue_script( 'groundhogg-logic-admin' );
	}

	/**
	 * Wrap the render function
	 *
	 * @param $funnel_step Funnel_Step
	 */
	public function wrap_render( $funnel_step ) {
		if ( ! $funnel_step->get_current_step() || $funnel_step->get_current_step()->is_logic() ) {
			return;
		}

		?>
        <div class="step-conditional-logic ignore-morph"></div>
		<?php
	}

	/**
	 * Wrapper to see if the contact passes the require logic.
	 *
	 * @param $logic   array
	 * @param $contact Contact
	 *
	 * @return bool
	 */
	protected function passes_logic( $logic, $contact ) {

		// Get the filters
		$include_filters = get_array_var( $logic, 'filters', [] );
		$exclude_filters = get_array_var( $logic, 'exclude_filters', [] );

		// No filters, return true
		if ( empty( $exclude_filters ) && empty( $include_filters ) ) {
			return true;
		}

		$query = new Contact_Query();

		// If this query returns 1, contact pass logic, otherwise fails.
		return $query->count( [
				'include'         => [ $contact->get_id() ],
				'filters'         => $include_filters,
				'exclude_filters' => $exclude_filters,
			] ) > 0;
	}


	/**
	 * Given a step that's about to be enqueued, filter it if the step based logic does not work
	 *
	 * @throws \Exception
	 *
	 * @param Step    $step    the step we're about to enqueue. Can't be strongly typed!!!!
	 * @param Contact $contact the contact record
	 *
	 * @return Step|false|null
	 */
	function filter_step( $step, Contact $contact ) {

		if ( ! is_a( $step, Step::class ) ) {
			return $step;
		}

		// Check if conditional logic is enabled
		$logic_enabled = boolval( $step->get_meta( '_conditional_logic_enabled' ) );

		// Not enabled? Return out;
		if ( ! $logic_enabled ) {
			return $step;
		}

		// Check if passes conditional logic
		$logic = $step->get_meta( '_conditional_logic' );

		// No Logic? Give up.
		if ( empty( $logic ) ) {
			return $step;
		}

		Simulator::log( '🧠 Evaluating step logic...' );

		// Hey! Go ahead
		if ( $this->passes_logic( $logic, $contact ) ) {
			Simulator::log( '☑️ Conditions met...' );

			return $step;
		}

		Simulator::log( '✖️ Conditions failed...' );

		// At this point, failed logic. If a benchmark, return false
		if ( $step->is_benchmark() ) {
			return null;
		}

		$result = get_array_var( $logic, 'result' );

		// handle failed logic for actions...
		switch ( $result ) {
			default:
			case 'skip':
				Simulator::log( '⏩ Skipping...' );

				return $step->get_next_action( $contact );
			case 'skip_to':
				$skip_to = absint( get_array_var( $logic, 'skip_to' ) );
				$next    = new Step( $skip_to );

				if ( $next->exists() ) {
					Simulator::log( '⏩ Skipping to target...' );

					return $next;
				}

				Simulator::log( '⏩ Skipping...' );

				return $step->get_next_action( $contact );

			case 'stop':
				Simulator::log( '🛑 Stopping the flow...' );

				// do nothing...
				return null;
		}
	}

}
