<?php
/**
 * Utility functions for the plugin.
 *
 * This file is for custom helper functions.
 * These should not be confused with WordPress template
 * tags. Template tags typically use prefixing, as opposed
 * to Namespaces.
 *
 * @link https://developer.wordpress.org/themes/basics/template-tags/
 * @package CommonFrameworkPlugin
 */

namespace CommonFrameworkPlugin\Utility;

/**
 * Get asset info from extracted asset files
 *
 * @param string $slug Asset slug as defined in build/webpack configuration
 * @param string $attribute Optional attribute to get. Can be version or dependencies
 * @return string|array
 */
function get_asset_info( $slug, $attribute = null ) {
	if ( file_exists( COMMON_FRAMEWORK_PLUGIN_PATH . 'dist/js/' . $slug . '.asset.php' ) ) {
		$asset = require COMMON_FRAMEWORK_PLUGIN_PATH . 'dist/js/' . $slug . '.asset.php';
	} elseif ( file_exists( COMMON_FRAMEWORK_PLUGIN_PATH . 'dist/css/' . $slug . '.asset.php' ) ) {
		$asset = require COMMON_FRAMEWORK_PLUGIN_PATH . 'dist/css/' . $slug . '.asset.php';
	} else {
		return null;
	}

	if ( ! empty( $attribute ) && isset( $asset[ $attribute ] ) ) {
		return $asset[ $attribute ];
	}

	return $asset;
}

/**
 * The list of knows contexts for enqueuing scripts/styles.
 *
 * @return array
 */
function get_enqueue_contexts() {
	return [ 'admin', 'frontend', 'shared' ];
}

/**
 * Generate an URL to a stylesheet, taking into account whether SCRIPT_DEBUG is enabled.
 *
 * @param string $stylesheet Stylesheet file name (no .css extension)
 * @param string $context Context for the script ('admin', 'frontend', or 'shared')
 *
 * @return string URL
 */
function style_url( $stylesheet, $context ) {

	if ( ! in_array( $context, get_enqueue_contexts(), true ) ) {
		return new WP_Error( 'invalid_enqueue_context', 'Invalid $context specified in CommonFrameworkPlugin stylesheet loader.' );
	}

	return COMMON_FRAMEWORK_PLUGIN_URL . "dist/css/{$stylesheet}.css";
}

/**
 * Generate an URL to a script, taking into account whether SCRIPT_DEBUG is enabled.
 *
 * @param string $script Script file name (no .js extension)
 * @param string $context Context for the script ('admin', 'frontend', or 'shared')
 *
 * @return string|WP_Error URL
 */
function script_url( $script, $context ) {

	if ( ! in_array( $context, get_enqueue_contexts(), true ) ) {
		return new WP_Error( 'invalid_enqueue_context', 'Invalid $context specified in CommonFrameworkPlugin script loader.' );
	}

	return COMMON_FRAMEWORK_PLUGIN_URL . "dist/js/{$script}.js";
}

/**
 * Sorts a multidimensional array by a specific key.
 *
 * @param array      $array The array to sort.
 * @param string|int $on The key or index to sort by.
 * @param int        $order The sort order. Accepts SORT_ASC (default) or SORT_DESC.
 *
 * @return array The sorted array.
 */
function array_sort( $array, $on, $order = SORT_ASC ) {
	$new_array = array();
	$sortable_array = array();

	if ( count( $array ) > 0 ) {
		foreach ( $array as $k => $v ) {
			if ( is_array( $v ) ) {
				foreach ( $v as $k2 => $v2 ) {
					if ( $k2 == $on ) {
						$sortable_array[ $k ] = $v2;
					}
				}
			} else {
				$sortable_array[ $k ] = $v;
			}
		}

		switch ( $order ) {
			case SORT_ASC:
				asort( $sortable_array );
				break;
			case SORT_DESC:
				arsort( $sortable_array );
				break;
		}

		foreach ( $sortable_array as $k => $v ) {
			$new_array[ $k ] = $array[ $k ];
		}
	}

	return $new_array;
}

/**
 * Retrieves all user emails and display names.
 *
 * @return array An associative array of user emails as keys and display names as values.
 */
function get_all_emails() {

	$all_users = get_users( [ 'fields' => [ 'user_email', 'display_name' ] ] );
	$user_email_list = [];

	foreach ( $all_users as $user ) {
		$user_email_list[ esc_html( $user->user_email ) ] = esc_html( $user->display_name );
	}

	return $user_email_list;
}
