<?php
/**
 * CommonFrameworkPlugin
 *
 * @package CommonFrameworkPlugin
 */

namespace CommonFrameworkPlugin;

use CommonFrameworkPlugin\Utility;

/**
 * Creates the CommonFrameworkPlugin post type and shortcodes to output them.
 */
class CommonFrameworkPlugin extends \CommonFrameworkPlugin\Module {
	/**
	 * Only register if on an admin page and if fieldmanager plugin is active.
	 *
	 * @return bool
	 */
	public function can_register() {
		return true;
	}

	/**
	 * Register our hooks.
	 *
	 * @return void
	 */
	public function register() {
		// Disables Pings during cron
		if ( isset( $_GET['doing_wp_cron'] ) ) {
			remove_action( 'do_pings', 'do_all_pings' );
			wp_clear_scheduled_hook( 'do_pings' );
		}

		if ( function_exists( 'acf_add_options_page' ) ) {
			acf_add_options_page(
				array(
					'page_title'     => 'Site Branding',
					'menu_title'    => 'Site Branding',
					'menu_slug'     => 'site-branding',
					'icon_url'      => 'dashicons-store',
					'redirect'        => false,
				)
			);
		}

		add_action( 'wp_enqueue_scripts', [ $this, 'enqueues' ] );
		add_filter( 'term_link', [ $this, 'get_term_link_product_edit' ], 10, 3 );
		add_filter( 'acf/format_value/type=textarea', [ $this, 'allow_shortcodes_in_textarea' ], 10, 3 );
		add_filter( 'upload_dir', [ $this, 'change_upload_directory_for_post_types' ] );
		add_filter( 'pre_get_posts', [ $this, 'change_post_count_for_post_types' ] );

		// remove filter that strips tags from the nav menu
		remove_filter( 'nav_menu_description', 'strip_tags' );

		add_filter( 'term_description', 'do_shortcode' );
		add_filter( 'category_description', 'do_shortcode' );
		add_filter( 'post_tag_description', 'do_shortcode' );
		add_filter( 'product-category_description', 'do_shortcode' );

		remove_filter( 'pre_term_description', 'wp_filter_kses' );
		remove_filter( 'pre_link_description', 'wp_filter_kses' );
		remove_filter( 'pre_link_notes', 'wp_filter_kses' );
		remove_filter( 'term_description', 'wp_kses_data' );
		remove_filter( 'product-category_description', 'wp_kses_data' );

		if ( function_exists( 'acf_add_local_field_group' ) ) {
			acf_add_local_field_group(
				array(
					'key' => 'group_62ce0dec2e6f2',
					'title' => 'Site Branding',
					'fields' => array(
						array(
							'key' => 'field_62ce0dfdfcd23',
							'label' => 'Company Name',
							'name' => 'company_name',
							'aria-label' => '',
							'type' => 'text',
							'instructions' => '',
							'required' => 0,
							'conditional_logic' => 0,
							'wrapper' => array(
								'width' => '',
								'class' => '',
								'id' => '',
							),
							'default_value' => '',
							'placeholder' => '',
							'prepend' => '',
							'append' => '',
							'maxlength' => '',
						),
						array(
							'key' => 'field_62cede8799762',
							'label' => 'Primary Email',
							'name' => 'primary_email',
							'aria-label' => '',
							'type' => 'email',
							'instructions' => '',
							'required' => 0,
							'conditional_logic' => 0,
							'wrapper' => array(
								'width' => '',
								'class' => '',
								'id' => '',
							),
							'default_value' => '',
							'placeholder' => '',
							'prepend' => '',
							'append' => '',
						),
						array(
							'key' => 'field_62cede9e99763',
							'label' => 'Fax Number',
							'name' => 'fax_number',
							'aria-label' => '',
							'type' => 'text',
							'instructions' => '',
							'required' => 0,
							'conditional_logic' => 0,
							'wrapper' => array(
								'width' => '',
								'class' => '',
								'id' => '',
							),
							'default_value' => '',
							'placeholder' => '',
							'prepend' => '',
							'append' => '',
							'maxlength' => '',
						),
						array(
							'key' => 'field_62cedeac99764',
							'label' => 'Phone Number',
							'name' => 'phone_number',
							'aria-label' => '',
							'type' => 'text',
							'instructions' => '',
							'required' => 0,
							'conditional_logic' => 0,
							'wrapper' => array(
								'width' => '',
								'class' => '',
								'id' => '',
							),
							'default_value' => '',
							'placeholder' => '',
							'prepend' => '',
							'append' => '',
							'maxlength' => '',
						),
						array(
							'key' => 'field_62cee08ea320c',
							'label' => 'Primary Address',
							'name' => 'primary_address',
							'aria-label' => '',
							'type' => 'text',
							'instructions' => '',
							'required' => 0,
							'conditional_logic' => 0,
							'wrapper' => array(
								'width' => '',
								'class' => '',
								'id' => '',
							),
							'default_value' => '',
							'placeholder' => '',
							'prepend' => '',
							'append' => '',
							'maxlength' => '',
						),
						array(
							'key' => 'field_62cee12825e67',
							'label' => 'Facebook Url',
							'name' => 'facebook_url',
							'aria-label' => '',
							'type' => 'url',
							'instructions' => '',
							'required' => 0,
							'conditional_logic' => 0,
							'wrapper' => array(
								'width' => '',
								'class' => '',
								'id' => '',
							),
							'default_value' => '',
							'placeholder' => '',
						),
						array(
							'key' => 'field_62cee10c25e66',
							'label' => 'Twitter Url',
							'name' => 'twitter_url',
							'aria-label' => '',
							'type' => 'url',
							'instructions' => '',
							'required' => 0,
							'conditional_logic' => 0,
							'wrapper' => array(
								'width' => '',
								'class' => '',
								'id' => '',
							),
							'default_value' => '',
							'placeholder' => '',
						),
						array(
							'key' => 'field_62cee14725e68',
							'label' => 'YouTube Url',
							'name' => 'youtube_url',
							'aria-label' => '',
							'type' => 'url',
							'instructions' => '',
							'required' => 0,
							'conditional_logic' => 0,
							'wrapper' => array(
								'width' => '',
								'class' => '',
								'id' => '',
							),
							'default_value' => '',
							'placeholder' => '',
						),
						array(
							'key' => 'field_62cee14e25e69',
							'label' => 'Linked In Url',
							'name' => 'linked_in_url',
							'aria-label' => '',
							'type' => 'url',
							'instructions' => '',
							'required' => 0,
							'conditional_logic' => 0,
							'wrapper' => array(
								'width' => '',
								'class' => '',
								'id' => '',
							),
							'default_value' => '',
							'placeholder' => '',
						),
						array(
							'key' => 'field_62d893e27470c',
							'label' => 'Emergency Service Phone Numbers',
							'name' => 'emergency_service_phone_numbers',
							'aria-label' => '',
							'type' => 'repeater',
							'instructions' => '',
							'required' => 0,
							'conditional_logic' => 0,
							'wrapper' => array(
								'width' => '',
								'class' => '',
								'id' => '',
							),
							'collapsed' => '',
							'min' => 0,
							'max' => 0,
							'layout' => 'table',
							'button_label' => 'Add Row',
							'sub_fields' => array(
								array(
									'key' => 'field_62d894077470d',
									'label' => 'Name',
									'name' => 'name',
									'aria-label' => '',
									'type' => 'text',
									'instructions' => '',
									'required' => 1,
									'conditional_logic' => 0,
									'wrapper' => array(
										'width' => '',
										'class' => '',
										'id' => '',
									),
									'default_value' => '',
									'placeholder' => '',
									'prepend' => '',
									'append' => '',
									'maxlength' => '',
									'parent_repeater' => 'field_62d893e27470c',
								),
								array(
									'key' => 'field_62d8940c7470e',
									'label' => 'Phone Number',
									'name' => 'phone_number',
									'aria-label' => '',
									'type' => 'text',
									'instructions' => '',
									'required' => 1,
									'conditional_logic' => 0,
									'wrapper' => array(
										'width' => '',
										'class' => '',
										'id' => '',
									),
									'default_value' => '',
									'placeholder' => '',
									'prepend' => '',
									'append' => '',
									'maxlength' => '',
									'parent_repeater' => 'field_62d893e27470c',
								),
							),
							'rows_per_page' => 20,
						),
						array(
							'key' => 'field_67d1749d1b5c1',
							'label' => 'White Logo',
							'name' => 'white_logo',
							'aria-label' => '',
							'type' => 'image',
							'instructions' => '',
							'required' => 0,
							'conditional_logic' => 0,
							'wrapper' => array(
								'width' => '',
								'class' => '',
								'id' => '',
							),
							'return_format' => 'array',
							'library' => 'all',
							'min_width' => '',
							'min_height' => '',
							'min_size' => '',
							'max_width' => '',
							'max_height' => '',
							'max_size' => '',
							'mime_types' => '',
							'allow_in_bindings' => 0,
							'preview_size' => 'medium',
						),
					),
					'location' => array(
						array(
							array(
								'param' => 'options_page',
								'operator' => '==',
								'value' => 'site-branding',
							),
						),
					),
					'menu_order' => 0,
					'position' => 'normal',
					'style' => 'default',
					'label_placement' => 'top',
					'instruction_placement' => 'label',
					'hide_on_screen' => '',
					'active' => true,
					'description' => '',
					'show_in_rest' => 0,
				)
			);
		}
	}

	/**
	 * Modify the term link for product categories.
	 *
	 * @param string $termlink The term link.
	 * @param object $term The term object.
	 * @param string $taxonomy The taxonomy name.
	 * @return string Modified term link.
	 */
	public function get_term_link_product_edit( $termlink, $term, $taxonomy ) {

		// Yoda condition to comply with WordPress coding standards.
		if ( 'product-category' === $taxonomy ) {
			$termlink = str_replace( '/product-category', '', $termlink );
		}

		return $termlink;
	}

	/**
	 * Allow shortcodes in ACF textarea fields.
	 *
	 * @param string $value The value of the field.
	 * @param int    $post_id The post ID.
	 * @param array  $field The field array.
	 * @return string The modified field value.
	 */
	public function allow_shortcodes_in_textarea( $value, $post_id, $field ) {
		$value = do_shortcode( $value );
		return $value;
	}

	/**
	 * Change the upload directory based on post type.
	 *
	 * @param array $args The upload directory arguments.
	 * @return array The modified upload directory arguments.
	 */
	public function change_upload_directory_for_post_types( $args ) {
		// Get the current post_id
		$id = isset( $_REQUEST['post_id'] ) ? sanitize_text_field( wp_unslash( $_REQUEST['post_id'] ) ) : '';

		if ( $id ) {
			$parent = get_post( $id )->post_parent;

			if ( 'page' === get_post_type( $id ) || 'post' === get_post_type( $id ) ) {
				return $args;
			}

			// Set the new path based on the current post type
			if ( 'white-papers' === get_post_type( $id ) || 'white-papers' === get_post_type( $parent ) ) {
				$newdir = '/' . get_post_type( $id );
				$args['path'] = $args['basedir'] . $newdir;
				$args['url']  = $args['baseurl'] . $newdir;
			}

			if ( 'video' === get_post_type( $id ) || 'video' === get_post_type( $parent ) ) {
				$newdir = '/videos';
				$args['path'] = $args['basedir'] . $newdir;
				$args['url']  = $args['baseurl'] . $newdir;
			}

			if ( 'products' === get_post_type( $id ) || 'products' === get_post_type( $parent ) ) {
				$newdir = '/' . get_post_type( $id );
				$args['path'] = $args['basedir'] . $newdir;
				$args['url']  = $args['baseurl'] . $newdir;
			}
		}

		return $args;
	}

	/**
	 * Change the post count for specific post types.
	 *
	 * @param object $query The WP_Query object.
	 * @return object The modified WP_Query object.
	 */
	public function change_post_count_for_post_types( $query ) {
		if ( is_admin() || ! $query->is_main_query() ) {
			return $query;
		}

		if ( is_post_type_archive( 'white-papers' ) || $query->is_search ) {
			$query->set( 'posts_per_page', 25 );
		}

		if ( is_post_type_archive( 'video' ) ) {
			$query->set( 'posts_per_page', 12 );
		}

		return $query;
	}

	/**
	 * Enqueue our built assets.
	 *
	 * @return void
	 */
	public function enqueues() {
		wp_enqueue_style(
			'common_framework_plugin_css',
			Utility\style_url( 'style', 'frontend' ),
			[],
			Utility\get_asset_info( 'style', 'version' )
		);
	}
}
