<?php
/**
 * CTAs
 *
 * @package CommonFrameworkPlugin
 */

namespace CommonFrameworkPlugin;

use CommonFrameworkPlugin\CPTGenerator;

/**
 * Creates the CommonFrameworkPlugin post type and shortcodes to output them.
 */
class CTAs extends \CommonFrameworkPlugin\Module {
	/**
	 * Only register if on an admin page and if fieldmanager plugin is active.
	 *
	 * @return bool
	 */
	public function can_register() {
		return true;
	}

	/**
	 * Register our hooks.
	 *
	 * @return void
	 */
	public function register() {
		add_action( 'init', [ $this, 'create_cta_custom_post_type' ], 10 );
		add_action( 'init', [ $this, 'create_custom_fields' ], 20 );
		// add_action( 'template_redirect', [ $this, 'disable_single_view' ], 10, 1 );
		add_filter( 'wpseo_robots', [ $this, 'exclude_from_yoast_index' ], 10, 2 );
		add_filter( 'wpseo_sitemap_exclude_post_type', [ $this, 'exclude_from_sitemap' ], 10, 2 );
	}

	/**
	 * Creates the custom post type for CTAs.
	 *
	 * @return void
	 */
	public function create_cta_custom_post_type() {
		$cpt = new CPTGenerator();

		$fields = [
			'slug'                  => 'cta',
			'singular'              => 'CTA',
			'plural'                => "CTA's",
			'menu_name'             => "CTA's",
			'description'           => "CTA's",
			'has_archive'           => false,
			'hierarchical'          => false,
			'menu_icon'             => 'dashicons-megaphone',
			'rewrite' => array(
				'slug'                  => 'cta',
				'with_front'            => false,
				'pages'                 => false,
				'feeds'                 => false,
				'ep_mask'               => EP_PERMALINK,
			),
			'menu_position'         => 24,
			'public'                => true,
			'publicly_queryable'    => true,
			'exclude_from_search'   => true,
			'show_ui'               => true,
			'show_in_menu'          => true,
			'query_var'             => true,
			'show_in_admin_bar'     => false,
			'show_in_nav_menus'     => false,
			'show_in_rest'          => true,
			'supports'              => array(
				'title',
				'editor',
				'excerpt',
				'author',
				'thumbnail',
				'custom-fields',
				'page-attributes',
			),
			'custom_caps'           => false,
			'custom_caps_users'     => array(
				'administrator',
			),
		];

		$cpt->register_single_post_type( $fields );
	}

	/**
	 * Creates custom fields for CTAs using Advanced Custom Fields (ACF).
	 *
	 * @return void
	 */
	public function create_custom_fields() {
		if ( function_exists( 'acf_add_local_field_group' ) ) {
			acf_add_local_field_group(
				array(
					'key' => 'group_64597b7ceb278',
					'title' => 'Footer CTA',
					'fields' => array(
						array(
							'key' => 'field_64597b7d3afa1',
							'label' => 'CTA',
							'name' => 'cta',
							'aria-label' => '',
							'type' => 'post_object',
							'instructions' => '',
							'required' => 0,
							'conditional_logic' => 0,
							'wrapper' => array(
								'width' => '',
								'class' => '',
								'id' => '',
							),
							'post_type' => array(
								0 => 'cta',
							),
							'taxonomy' => '',
							'return_format' => 'id',
							'multiple' => 0,
							'allow_null' => 0,
							'ui' => 1,
							'bidirectional_target' => array(),
						),
					),
					'location' => array(
						array(
							array(
								'param' => 'post_status',
								'operator' => '!=',
								'value' => 'trash',
							),
						),
					),
					'menu_order' => 0,
					'position' => 'side',
					'style' => 'default',
					'label_placement' => 'top',
					'instruction_placement' => 'label',
					'hide_on_screen' => '',
					'active' => true,
					'description' => '',
					'show_in_rest' => 1,
				)
			);
		}
	}

	/**
	 * Disable the single view of 'cta' post type and redirect to the archive page.
	 *
	 * This function checks if the current query is not for the admin panel, is the main query,
	 * and is for a singular 'cta' post type. If all conditions are met, it redirects
	 * the user to the archive page of 'cta'.
	 *
	 * @param WP_Query $query The current WordPress query instance.
	 *
	 * @return void
	 */
	public function disable_single_view( $query ) {
		if ( ! is_admin() && is_main_query() && is_singular( 'cta' ) ) {
			wp_safe_redirect( get_post_type_archive_link( 'cta' ) );
			exit;
		}
	}

	/**
	 * Modify Yoast SEO robots meta to exclude 'cta' post type from indexing.
	 *
	 * This function checks if the current post is a singular 'cta' post type,
	 * and if so, sets the 'noindex' directive in the robots meta for Yoast SEO.
	 *
	 * @param array   $robots An array of robots meta directives.
	 * @param WP_Post $post   The current post object.
	 *
	 * @return array Modified robots meta directives.
	 */
	public function exclude_from_yoast_index( $robots, $post ) {
		if ( is_singular( 'cta' ) ) {
			if ( is_array( $robots ) ) {
				$robots['index'] = 'noindex';
			} else {
				$robots = 'noindex';
			}
		}
		return $robots;
	}

	/**
	 * Exclude a post type from XML sitemaps.
	 *
	 * @param boolean $excluded  Whether the post type is excluded by default.
	 * @param string  $post_type The post type to exclude.
	 *
	 * @return bool Whether a given post type should be excluded.
	 */
	public function exclude_from_sitemap( $excluded, $post_type ) {
		return 'cta' === $post_type;
	}
}
