<?php
/**
 * BuyersGuides
 *
 * @package CommonFrameworkPlugin
 */

namespace CommonFrameworkPlugin;

use setasign\Fpdi\Fpdi;

/**
 * Creates the CommonFrameworkPlugin post type and shortcodes to output them.
 */
class BuyersGuides extends \CommonFrameworkPlugin\Module {
	/**
	 * Only register if on an admin page and if fieldmanager plugin is active.
	 *
	 * @return bool
	 */
	public function can_register() {
		return true;
	}

	/**
	 * Register our hooks.
	 *
	 * @return void
	 */
	public function register() {
		add_action( 'admin_menu', [ $this, 'add_cf_tools_page' ] );
		add_action( 'admin_post_create_dynamic_pdfs', [ $this, 'create_dynamic_pdfs' ] );
	}

	/**
	 * Adds network admin page which will be used to import products.
	 */
	public function add_cf_tools_page() {
		if ( empty( $GLOBALS['admin_page_hooks']['cf_tools'] ) ) {
			add_menu_page(
				'CF Tools',
				'CF Tools',
				'manage_options',
				'cf_tools',
				[ $this, 'cf_tools_page' ],
				'dashicons-admin-network',
				2
			);
		}
	}

	/**
	 * Displays the data on the Common Framework Network Admin Page
	 */
	public function cf_tools_page() {
		include COMMON_FRAMEWORK_PLUGIN_INC . 'pages/tools.php';
	}

	/**
	 * Creates dynamic PDFs based on various sources.
	 *
	 * @return void
	 */
	public function create_dynamic_pdfs() {
		$pdf_directory = COMMON_FRAMEWORK_PLUGIN_PATH . '/pdfs/';
		$ups_file_name = 'ups-buyers-guide.pdf';
		$pdu_file_name = 'pdu-buyers-guide.pdf';
		$edge_file_name = 'edge-buyers-guide.pdf';
		$rack_file_name = 'rack-systems-buyers-guide.pdf';
		$thermal_file_name = 'thermal-management-buyers-guide.pdf';
		$inter_sol_file_name = 'integrated-solutions-buyers-guide.pdf';

		$this->create_pdf_from_source( $pdf_directory . $ups_file_name, 'ups-buyers-guide' );
		$this->create_pdf_from_source( $pdf_directory . $pdu_file_name, 'pdu-buyers-guide' );
		$this->create_pdf_from_source( $pdf_directory . $edge_file_name, 'edge-buyers-guide' );
		$this->create_pdf_from_source( $pdf_directory . $rack_file_name, 'rack-systems-buyers-guide' );
		$this->create_pdf_from_source( $pdf_directory . $thermal_file_name, 'thermal-management-buyers-guide' );
		$this->create_pdf_from_source( $pdf_directory . $inter_sol_file_name, 'integrated-solutions-buyers-guide' );

		// Exits admin_post action when manually running the cron from the admin page.
		if ( ! empty( $_POST['action'] ) && 'create_dynamic_pdfs' === $_POST['action'] ) {
			$redirect_url = get_site_url() . '/wp-admin/admin.php?page=cf_tools';
			wp_safe_redirect( $redirect_url );
			exit;
		}
	}

	/**
	 * Creates a PDF from a source file.
	 *
	 * @param string $source_file Path to the source file.
	 * @param string $new_file_name Name of the new PDF file.
	 * @return void
	 */
	public function create_pdf_from_source( $source_file, $new_file_name ) {
		if ( ! defined( 'FPDF_FONTPATH' ) ) {
			define( 'FPDF_FONTPATH', COMMON_FRAMEWORK_PLUGIN_PATH . 'fonts/' );
		}

		// Sets, checks, and creates directory for product documents
		$upload_dir = wp_upload_dir();
		$document_directory = $upload_dir['basedir'] . '/buyers-guides';
		$document_directory_uri = $document_directory . '/';

		if ( ! file_exists( $document_directory ) ) {
			wp_mkdir_p( $document_directory );
		}

		$this->create_pdf( $document_directory_uri, $source_file, $new_file_name );
	}

	/**
	 * Creates the PDF.
	 *
	 * @param string   $document_directory_uri URI of the document directory.
	 * @param string   $source_file Path to the source file.
	 * @param string   $new_file_name Name of the new file.
	 * @param int|null $blog_id Blog ID if multisite.
	 * @return void
	 */
	private function create_pdf( $document_directory_uri, $source_file, $new_file_name, $blog_id = null ) {
		// Site Logo
		$custom_logo_id = get_theme_mod( 'custom_logo' );
		$image = [];
		if ( $custom_logo_id ) {
			$image = wp_get_attachment_image_src( $custom_logo_id, 'full' );
		} else {
			$image[0] = 'https://placehold.co/50x50.png';
		}

		$pdf = new Fpdi();
		$page_count = $pdf->setSourceFile( $source_file );
		$pdf->AddFont( 'Lato-Regular', 'R', 'Lato-Regular.php' );
		$pdf->SetFont( 'Lato-Regular', 'R', 18 );

		for ( $page_no = 1; $page_no <= $page_count; $page_no++ ) {
			$template_id = $pdf->importPage( $page_no );
			$pdf->AddPage();
			$pdf->useTemplate( $template_id, [ 'adjustPageSize' => true ] );

			if ( 1 === $page_no ) { // Yoda condition
				$pdf->Image(
					$image[0], // File
					10,  // x
					10,  // y
					50, // w
					0,  // h
					'',  // type
					home_url(), // link
					'center', // align
					false, // resize
					300 // dpi
				);
			}
		}

		$blog_id = is_null( $blog_id ) ? 1 : $blog_id;
		$pdf->Output( 'F', $document_directory_uri . $new_file_name . '-' . $blog_id . '.pdf' );
	}
}
