<?php
/**
 * DistributorCustomizations
 *
 * @package CommonFrameworkPlugin
 */

namespace CommonFrameworkPlugin;

use Distributor\ExternalConnection;
use WP_CLI;
use WP_Query;

/**
 * Customizes the behavior of the Distributor plugin by handling post meta whitelisting,
 * preserving parent-child relationships during syndication, managing cache and styling updates
 * post-syndication, and ensuring remote media is properly localized.
 */
class DistributorCustomizations extends \CommonFrameworkPlugin\Module {

	/**
	 * Distributor helper instance.
	 *
	 * @var DistributorHelper
	 */
	protected $helper;

	/**
	 * Constructor.
	 * Requires the Distributor helper class and initializes the helper property.
	 *
	 * @return void
	 */
	public function __construct() {
		require_once COMMON_FRAMEWORK_PLUGIN_INC . 'helpers/distributor.php';
		$this->helper = new DistributorHelper();
	}

	/**
	 * Only register if on an admin page and if the Fieldmanager plugin is active.
	 *
	 * @return bool True if the registration is allowed.
	 */
	public function can_register() {
		return true;
	}

	/**
	 * Register our hooks.
	 *
	 * @return void
	 */
	public function register() {
		add_action( 'admin_init', [ $this, 'setup_defaults' ], 99, 3 );

		// Initial push and pull, already correct for first distribution
		add_filter( 'dt_push_post_args', [ $this, 'handle_push_post' ], 10, 4 ); // Format for initial push on source
		add_filter( 'dt_post_to_pull', [ $this, 'format_post_for_pull' ], 10, 1 ); // Format for initial pull on source

		add_filter( 'dt_pull_post_args', [ $this, 'handle_pull_post' ], 10, 4 ); // Ingestion of pull on receiver
		add_action( 'dt_pull_post', [ $this, 'after_receive_any' ], 10, 3 ); // Ingestion of push/pull updates on receiver

		// Every later save that updates subscribers
		add_filter( 'dt_subscription_post_args', [ $this, 'subscription_payload' ], 10, 2 ); // Format for subscription push/pull
		add_action( 'dt_process_subscription_attributes', [ $this, 'on_subscription_receive' ], 100, 2 ); // Ingestion of subscription push/pull

		add_action( 'init', [ $this, 'hook_into_all_post_types' ] ); // Ingestion of initial push on receiver
		// Add header to remote_post during push so receiver knows what hook to ingest
		add_filter(
			'dt_auth_format_post_args',
			function ( $args, $context, $connection ) {
				$args['headers']['X-Distributor-Request'] = 'TRUE';
				return $args;
			},
			10,
			3
		);

		add_filter( 'post_row_actions', [ $this, 'filter_admin_row_actions' ], 100, 2 );
		add_filter( 'page_row_actions', [ $this, 'filter_admin_row_actions' ], 100, 2 );

		add_action(
			'elementor/editor/init',
			function () {
				$post_id = get_the_ID();
				if ( get_post_meta( $post_id, 'dt_syndicate_time', true ) ) {
					if ( is_admin() && ! defined( 'DOING_AJAX' ) ) {
						wp_die( 'This post is syndicated and cannot be edited with Elementor.' );
					}
				}
			},
			10
		);

		// TODO: Auto remove subscription and unlink post when deleted.
		// add_action( 'before_delete_post', [ $this, 'unlink_distributor_post_on_delete' ] );
		add_filter( 'dt_auto_distribution_enabled', '__return_true' );
		add_filter(
			'auto_distribute_supported_post_types',
			function ( $post_types ) {
				return [ 'products' ];
			}
		);

		add_filter(
			'dt_push_post_timeout',
			function ( $timeout, $post ) {
				return 90;
			},
			10,
			2
		);

		if ( defined( 'WP_CLI' ) && WP_CLI ) {
			WP_CLI::add_command(
				'distributor distribute-products',
				function () {
					$this->distribute_all_products();
				}
			);
		}
	}

	/**
	 * Sets default configuration values for the plugin and stores them in the distributor_settings WordPress option.
	 *
	 * @return void
	 */
	public function setup_defaults() {
		$this->helper::setup_defaults();
	}

	/**
	 * Modifies post arguments before pushing to ensure the data is included for pages.
	 *
	 * @param array              $post_body The request body to send.
	 * @param \WP_Post           $post The WP_Post that is being pushed.
	 * @param array              $args Post args to push.
	 * @param ExternalConnection $connection The distributor connection being pushed to.
	 * @return array The post data to be inserted.
	 */
	public function handle_push_post( $post_body, $post, $args, $connection ) {
		write_log( 'handling formatting for push' );
		$transport = $this->helper::get_transport_for_connection( $connection ); // external or internal
		$payload   = $this->helper::build_common_payload( $post );
		$post_body = array_merge( $post_body, $this->helper::adapt_for_transport( $payload, $transport ) );
		return $post_body;
	}

	/**
	 * Format a post's data for pull operations.
	 * Merges the Distributor common payload into the provided display data.
	 *
	 * @param array $display_data Post data array provided by Distributor before pulling.
	 * @return array Modified post data array including canonical payload fields.
	 */
	public function format_post_for_pull( $display_data ) {
		$post = get_post( $display_data['ID'] );
		$payload   = $this->helper::build_common_payload( $post );
		$display_data = custom_array_merge( $display_data, $payload );
		return $display_data;
	}

	/**
	 * Ensures data is preserved when pulling content from another site.
	 *
	 * @param array              $post_array The post data to be inserted.
	 * @param array              $remote_post_id The remote post ID.
	 * @param \WP_Post           $post The request that got the post.
	 * @param ExternalConnection $connection The Distributor connection pulling the post.
	 * @return array The post data to be inserted.
	 */
	public function handle_pull_post( $post_array, $remote_post_id, $post, $connection ) {
		write_log( 'handling pull post' );

		if ( isset( $post->post_date ) ) {
			$post_array['post_date'] = $post->post_date;
		}
		if ( isset( $post->post_date_gmt ) ) {
			$post_array['post_date_gmt'] = $post->post_date_gmt;
		}

		$incoming = is_array( $post ) ? $post : (array) $post;
		foreach ( [ 'acf_post_fields', 'acf_term_fields', 'acf_post_fields_k', 'acf_term_fields_k', 'parent_ref', 'date', 'date_gmt', 'term_hierarchy', 'term_props', 'acf_relations' ] as $k ) {
			if ( isset( $incoming[ $k ] ) ) {
				$post_array[ $k ] = $incoming[ $k ];
			}
		}

		return $post_array;
	}

	/**
	 * Source site: add custom data to every subscription update.
	 * Runs when Distributor sends updates after you save a post.
	 *
	 * @param array    $post_body The post data being sent.
	 * @param \WP_Post $post The post being sent.
	 * @return array The updated post data.
	 */
	public function subscription_payload( $post_body, $post ) {
		write_log( 'sending subscription payload' );
		$payload = $this->helper::build_common_payload( $post );
		return array_merge( $post_body, $this->helper::adapt_for_transport( $payload, 'subscription' ) );
	}

	/**
	 * Receiver site: apply subscription payload, then clear caches and regenerate CSS.
	 * Runs on every subscriber update.
	 *
	 * @param \WP_Post         $post The post being updated.
	 * @param \WP_REST_Request $request The request containing the payload.
	 * @return void
	 */
	public function on_subscription_receive( \WP_Post $post, \WP_REST_Request $request ) {
		write_log( 'receiving subscription payload' );
		$params = $request->get_params();
		$this->helper::apply_incoming_payload( $post->ID, $params );
		$this->helper::localize_elementor_media( $post->ID );
		$this->helper::after_content_update_caches( $post->ID );
	}

	/**
	 * Hook into all registered post types.
	 * Adds rest_after_insert_* hooks for each post type so payloads can be applied on insert.
	 *
	 * @return void
	 */
	public function hook_into_all_post_types() {
		// Get all registered post types
		$post_types = get_post_types( [], 'names' );

		// Loop through each post type and add the rest_after_insert hook dynamically
		foreach ( $post_types as $post_type ) {
			add_action( "rest_after_insert_{$post_type}", [ $this, 'apply_payload_on_rest_insert' ], 10, 3 );
		}
	}

	/**
	 * Apply a Distributor payload to a post immediately after it is inserted via the REST API.
	 * Only runs when the request contains the X-Distributor-Request header, which signals an incoming push.
	 * Delegates handling to on_subscription_receive().
	 *
	 * @param \WP_Post         $post     The newly created or updated post object.
	 * @param \WP_REST_Request $request  The REST API request object.
	 * @param bool             $creating Whether the post is being created (true) or updated (false).
	 * @return void
	 */
	public function apply_payload_on_rest_insert( $post, $request, $creating ) {
		$from_distributor = $request->get_header( 'X-Distributor-Request' );
		if ( ! $from_distributor ) {
			return;
		}
		write_log( 'Getting an initial push on remote site' );
		$this->on_subscription_receive( $post, $request );
	}

	/**
	 * Receiver site: also clear caches after the first pull.
	 *
	 * @param int   $new_post_id The ID of the newly received post.
	 * @param mixed $connection The connection that received the post.
	 * @param array $post_array The post data that was received.
	 * @return void
	 */
	public function after_receive_any( $new_post_id, $connection, $post_array ) {
		write_log( 'running after any received payload' );
		$this->helper::apply_incoming_payload( $new_post_id, $post_array );
		$this->helper::localize_elementor_media( $new_post_id );
		$this->helper::after_content_update_caches( $new_post_id );
	}

	/**
	 * Add/Remove edit link in dashboard.
	 *
	 * Add or remove an edit link to the post/page action links on the post/pages list table.
	 *
	 * Fired by `post_row_actions` and `page_row_actions` filters.
	 *
	 * @access public
	 *
	 * @param array    $actions An array of row action links.
	 * @param \WP_Post $post The current post or page object.
	 *
	 * @return array An updated array of row action links.
	 */
	public function filter_admin_row_actions( $actions, $post ) {
		if ( isset( $actions['edit_with_elementor'] ) && get_post_meta( $post->ID, 'dt_syndicate_time', true ) && ! get_post_meta( $post->id, 'dt_unlinked', true ) ) {
			unset( $actions['edit_with_elementor'] );
		}

		return $actions;
	}

	/**
	 * Distribute all products that have not yet been syndicated.
	 *
	 * ## EXAMPLES
	 *
	 *     wp distributor distribute-products
	 *
	 * @when after_wp_load
	 */
	public function distribute_all_products() {

		if ( ! function_exists( '\Distributor\AutoDistribute\distribute_post_to_all_connections' ) ) {
			WP_CLI::error( 'Distributor plugin not loaded or AutoDistribute namespace missing.' );
		}

		$post_type = 'products';
		$query     = new WP_Query(
			[
				'post_type'      => $post_type,
				'post_status'    => [ 'any' ],
				'posts_per_page' => -1,
				'fields'         => 'ids',
			]
		);

		$total = count( $query->posts );

		if ( 0 === $total ) {
			WP_CLI::success( 'No products found to distribute.' );
			return;
		}

		WP_CLI::log( "Found {$total} products to distribute..." );

		$user_id = get_current_user_id() ? get_current_user_id() : 1;

		$distributed = 0;
		$errored = 0;
		$skipped = 0;

		foreach ( $query->posts as $post_id ) {
			$post = get_post( $post_id );

			// Skip if already has a connection map
			$map = get_post_meta( $post_id, 'dt_connection_map', true );
			if ( ( ! empty( $map['internal'] ) && 20 !== count( $map['internal'] ) ) || ( ! empty( $map['external'] ) && 20 !== count( $map['external'] ) ) ) {
				$skipped++;
				WP_CLI::log( "Skipping product ID {$post_id}: already distributed." );
				continue;
			}

			try {

				\Distributor\AutoDistribute\distribute_post_to_all_connections( $post, $user_id );
				$distributed++;
				WP_CLI::log( "Distributed product ID {$post_id}" );
			} catch ( \Throwable $e ) {
				$errored++;
				WP_CLI::warning( "Failed to distribute product ID {$post_id}: " . $e->getMessage() );
			}
		}

		WP_CLI::success( "Distribution complete. {$distributed} products distributed, skipped {$skipped} products, and {$errored} products failed." );
	}
}
