<?php
/**
 * Service Provider for Linked_Posts functionality.
 *
 * @since 6.2.0
 *
 * @package TEC\Events_Pro\Linked_Posts
 */

namespace TEC\Events_Pro\Linked_Posts;

use TEC\Common\Contracts\Provider\Controller as Controller_Contract;
use TEC\Events_Pro\Linked_Posts\Venue;


use WP_Post;

/**
 * Class Provider
 *
 * @since 6.2.0

 * @package TEC\Events_Pro\Linked_Posts
 */
class Controller extends Controller_Contract {
	/**
	 * Register the controller.
	 *
	 * @since 6.2.0
	 */
	public function do_register(): void {
		$this->container->singleton( Thumbnail_Inclusion::class );
		$this->container->singleton( Coordinates_Inclusion::class );
		$this->container->register( Organizer\Controller::class );
		$this->container->register( Venue\Controller::class );

		$this->add_actions();
		$this->add_filters();
	}

	/**
	 * Unregister the controller.
	 *
	 * @since 6.2.0
	 *
	 * @return void
	 */
	public function unregister(): void {
		$this->remove_actions();
		$this->remove_filters();
	}

	/**
	 * Add the filter hooks.
	 *
	 * @since 6.2.0
   */
	public function add_actions(): void {

	}

	/**
	 * Remove the action hooks.
	 *
	 * @since 6.2.0
	 */
	public function remove_actions(): void {

	}

	/**
	 * Unregister the controller.
	 *
	 * @since 6.2.0
	 */
	public function add_filters(): void {
		add_filter( 'tribe_get_organizer_object', [ $this, 'include_thumbnail_in_object' ], 15, 3 );
		add_filter( 'tribe_get_venue_object', [ $this, 'include_thumbnail_in_object' ], 15, 3 );
		add_filter( 'tribe_get_venue_object', [ $this, 'include_coordinates_in_object' ], 15 );
		add_filter( 'tec_rest_venue_properties_to_add', [ $this, 'add_coordinates_to_rest_properties' ] );
		add_filter( 'admin_post_thumbnail_html', [ $this, 'include_helper_text_post_metabox' ], 15 );
	}

	/**
	 * Remove the filter hooks.
	 *
	 * @since 6.2.0
	 */
	public function remove_filters(): void {
		remove_filter( 'tribe_get_organizer_object', [ $this, 'include_thumbnail_in_object' ], 15 );
		remove_filter( 'tribe_get_venue_object', [ $this, 'include_thumbnail_in_object' ], 15 );
		remove_filter( 'tribe_get_venue_object', [ $this, 'include_coordinates_in_object' ], 15 );
		remove_filter( 'tec_rest_venue_properties_to_add', [ $this, 'add_coordinates_to_rest_properties' ] );
		remove_filter( 'admin_post_thumbnail_html', [ $this, 'include_helper_text_post_metabox' ], 15 );
	}

	/**
	 * Include the thumbnail in an object of Linked Post.
	 *
	 * @since 6.2.0
	 *
	 * @param WP_Post $post   The organizer post object, decorated with a set of custom properties.
	 * @param string  $output The output format to use.
	 * @param string  $filter The filter, or context of the fetch.
	 *
	 * @return mixed
	 */
	public function include_thumbnail_in_object( $post, $output, $filter ) {
		if ( ! $post instanceof WP_Post ) {
			return $post;
		}
		return $this->container->make( Thumbnail_Inclusion::class )->include_thumbnail( $post, $output, $filter );
	}

	/**
	 * Include the coordinates in a Venue object.
	 *
	 * @since 7.7.0
	 *
	 * @param WP_Post $post The venue post object, decorated with a set of custom properties.
	 *
	 * @return mixed
	 */
	public function include_coordinates_in_object( $post ) {
		if ( ! $post instanceof WP_Post ) {
			return $post;
		}

		return $this->container->make( Coordinates_Inclusion::class )->include_coordinates( $post );
	}

	/**
	 * Filters the `admin_post_thumbnail_html` to add image aspect ratio recommendation.
	 *
	 * @since 6.2.0
	 *
	 * @param string $html The HTML for the featured image box.
	 *
	 * @return string The modified html, if required.
	 */
	public function include_helper_text_post_metabox( string $html ): string {
		return $this->container->make( Thumbnail_Inclusion::class )->include_helper_text_post_metabox( $html );
	}

	/**
	 * Adds latitude and longitude slugs to the REST venue properties list.
	 *
	 * @since 7.7.0
	 *
	 * @param array<string,bool> $properties The properties already slated for inclusion.
	 *
	 * @return array<string,bool>
	 */
	public function add_coordinates_to_rest_properties( array $properties ): array {
		/** @var Coordinates_Inclusion $coordinates_inclusion */
		$coordinates_inclusion = $this->container->make( Coordinates_Inclusion::class );
		return $coordinates_inclusion->filter_rest_properties( $properties );
	}
}
